local socket = require('socket')

-- Thread management values
local threads = {}
local current_thread = nil
local sbFlushImage = image(1,1)
local async_queue = {}

-- Thread management task
local managementTask = tween.interval(0.001, function()
    
    -- Reset per frame flags
    for _,thread in ipairs(threads) do
        thread.done_this_frame = false
    end
    
    -- Keep running while threads are still executing
    local thread_ran = true
    while thread_ran do
        
        -- Reset flag
        thread_ran = false
        
        local deadThreads = {}
    
        for i,thread in ipairs(threads) do
            
            local dbg_thread = Debugger.thread
            
            -- Is the coroutine dead?
            if coroutine.status(thread.co) == "dead" then
                table.insert(deadThreads, i)
                
            -- Has the thread finished for this frame?
            elseif thread.done_this_frame then
                
            -- Ignore this thread if it's sleeping
            elseif thread.sleep_end and socket.gettime() < thread.sleep_end then
                
                -- If the sleeping thread is 'blocking' pretend we ran
                if thread.blocking then
                    thread_ran = true
                end
                
            -- If we're debugging, only run the debugged thread
            elseif dbg_thread ~= nil and thread ~= dbg_thread then
                
            -- All checks have passed. Resume the thread!
            else
                -- Track the current thread
                current_thread = thread
                thread_ran = true
                
                local runMainRet = {}
                while true do
                    local res = table.pack(coroutine.resume(thread.co, table.unpack(runMainRet)))
                    if not res[1] then
                        -- Propagate the error
                        error(string.format("\n%s\n\n%s\n\nIGNORE TRACE BELOW:", res[2], debug.traceback(thread.co)))
                    elseif res[2] then
                        -- Call the value returned from the resume
                        -- and resume again.
                        -- This is how a thread forces something to
                        -- run on the main thread.
                        current_thread = nil
                        runMainRet = table.pack(res[2](select(3, table.unpack(res))))
                        current_thread = thread
                    else
                        -- Pause execution
                        break
                    end
                end
                
                current_thread = nil
            end
        end
        
        -- Remove dead threads
        for i=#deadThreads,1,-1 do
            table.remove(threads, deadThreads[i])
        end
    end
        
    -- Render an offscreen sprite to flush the
    -- spritebatcher.
    -- This ensures sprite rendering done on
    -- threads is actually visible on screen.
    -- A wild guess is that the batcher is only
    -- flushed after the draw() call, and not
    -- after tween handling.
    spriteMode(CORNER)
    sprite(sbFlushImage, -1, -1)
end)
managementTask.noAsyncCallback = true
managementTask.ignoreDebug = true

-- Thread functions
function yield(fast)
    if not coroutine.isyieldable() then
        error("Unable to yield!")
    end
    current_thread.done_this_frame = (fast == nil or fast == false)
    coroutine.yield()
end

function yieldframe()
    yield()
end


-- Thread class
Thread = class()
Thread.__name = "Codea+ Thread"
function Thread:init(fn, name)
    self.co = coroutine.create(fn)
    self.blocking = false
    self.name = name
    
    table.insert(threads, self)
end

function Thread.runOnMain(func, ...)
    if coroutine.isyieldable() then
        return coroutine.yield(func, ...)
    end
    func(...)
end

function Thread.__runAsync(func, ...)
    if func == nil or Debugger.thread then return end
    local params = {...}
    for i,p in ipairs(params) do
        -- Convert userdata param to a table
        if type(p) == "userdata" and p.___getters then
            local np = {}
            for k,fn in pairs(p.___getters) do
                np[k] = fn(p)
            end
            params[i] = np
        end
    end
    table.insert(async_queue, { 
        fn = func,
        params = params
    })
end

-- Threaded callbacks
function Thread.callback(fn, ignoreDebugger)
    return function(...)
        if fn == nil or (Debugger.thread and not ignoreDebugger) then return end
        table.insert(async_queue, { 
            fn = fn,
            params = { ... }
        })
    end
end

function Thread.wrapMain(fn)
    return function(...)
        return Thread.runOnMain(fn, ...)
    end
end

function Thread.current()
    return current_thread
end

function Thread.allThreads()
    return threads
end

function Thread.sleep(duration)
    current_thread.sleep_end = socket.gettime() + duration
    yield()
end

__wrapGlobal("touched", Thread.__runAsync)
__wrapGlobal("hover", Thread.__runAsync)
__wrapGlobal("scroll", Thread.__runAsync)
__wrapGlobal("pinch", Thread.__runAsync)
__wrapGlobal("keyboard", Thread.__runAsync)
-- __wrapGlobal("willClose", Thread.__runAsync) -- Don't wrap this

-- Async callback thread
Thread(function()
    
    -- We're a blocking thread as callbacks should
    -- be handled in the same frame as they were issued
    current_thread.blocking = true
    
    while true do
        for _,cb in ipairs(async_queue) do
            cb.fn(table.unpack(cb.params))
        end
        async_queue = {}
        
        yieldframe()
    end
    
end)

-- These should do nothing
__wrapGlobal("setup", function() end)
__wrapGlobal("draw", function() end)

-- Thread to call setup() & draw()
Thread(function()
    
    -- We're a blocking thread
    current_thread.blocking = true
    
    -- Call setup() once
    local fn = __getWrappedGlobal("setup")
    if fn then fn() end
    
    -- Call draw() repeatedly
    while true do
        local fn = __getWrappedGlobal("draw")
        if fn then fn() end
        yieldframe()
    end
end)