-- Some of Codea's builtin functions or objects
-- work incorrectly from within a coroutine.
-- The functions below hijack the metatable's of such
-- objects to ensure that they are only accessed from
-- the main thread (not a coroutine)

local wrapFunction = nil
local wrapTable = nil
local wrapUserdata = nil

local function wrap(lazy, ...)
    local res = {}
    
    for _,obj in ipairs({...}) do
        local t = type(obj)
        if obj == _G then -- Don't wrap the global table
            table.insert(res, obj)
        elseif t == "function" then
            table.insert(res, wrapFunction(obj))
        elseif t == "table" then
            table.insert(res, wrapTable(obj, lazy))
        elseif t == "userdata" then
            table.insert(res, wrapUserdata(obj))
        else
            table.insert(res, obj)
        end
    end
    
    return table.unpack(res)
end

wrapFunction = function(fn)
    return Thread.wrapMain(fn)
end

wrapTable = function(t, lazy)
    
    if lazy then
        
        -- Original metatable
        local omt = getmetatable(t)
        
        -- If we've already been converted, return table
        if omt and omt.___mainAccess then return t end
        
        -- Create a shallow clone of the table
        -- and clear the original to force it into
        -- our new metatable below
        local clone = {}
        for k, v in pairs(t) do
            clone[k] = v
            rawset(t, k, nil)
        end
        for k, v in ipairs(t) do
            clone[k] = v
            rawset(t, k, nil)
        end
        
        -- If the original metatable IS the table,
        -- the clone should do the same.
        if omt == t then
            setmetatable(clone, clone)
            omt = clone -- Consider the clone the new original mt
        else
            setmetatable(clone, omt)
        end
        
        -- Our new metatable
        local mt = {
            ___lazyObj = clone,
            ___mainAccess = true,
            
            __index = Thread.wrapMain(function(t, k)
                return wrap(true, clone[k])
            end),
            
            __newindex = Thread.wrapMain(function(t, k, v)
                clone[k] = v
            end),
            
            
        }
        
        setmetatable(t, mt)
    else
        
        -- Get our metatable
        local mt = getmetatable(t)
        
        -- Get our lazy object
        local lazyObj = mt.___lazyObj
        
        -- Copy all the lazy object values back
        -- into our table (wrapped this time)
        for k, v in pairs(lazyObj) do
            rawset(t, k, wrap(true, v))
        end
        for k, v in ipairs(lazyObj) do
            rawset(t, k, wrap(true, v))
        end
        
        -- We don't need our lazy object any more
        mt.___lazyObj = nil
    end
    
    return t
end

wrapUserdata = function(ud)
    local mt = getmetatable(ud)
            
    -- No metatable? Just add the userdata obj
    -- or already converted?
    if mt == nil or rawget(mt, "___mainAccess") == true then return ud end
    
    -- Set the 'converted' flag            
    rawset(mt, "___mainAccess", true)
    
    -- Custom __index function
    local __index = mt.__index
    mt.__index = function(t, k)
        -- '__cache' gets special non-wrapped treatment
        -- due to some Codea internals likely accessing
        -- its values directly using rawget.
        if k == "__cache" then
            return __index(t, k)
        end
        
        return wrap(true, Thread.runOnMain(__index, t, k))
    end
    
    -- Wraps the named metamethod in order to force it
    -- to execute on the main thread.
    local function wrapmm(name)
        local mm = rawget(mt, name)
        if type(mm) == "function" then
            -- Force it to execute on the main thread
            -- & ensure return values are also wrapped
            rawset(mt, name, function(...)
                return wrap(true, Thread.runOnMain(mm, ...))
            end)
        elseif mm then
            -- Ensure the value is adequately thread safe
            rawset(mt, name, wrap(true, mm))
        end
    end
    
    wrapmm("__add")
    wrapmm("__sub")
    wrapmm("__mul")
    wrapmm("__div")
    wrapmm("__mod")
    wrapmm("__pow")
    wrapmm("__unm")
    wrapmm("__idiv")
    wrapmm("__band")
    wrapmm("__bor")
    wrapmm("__bxor")
    wrapmm("__bnot")
    wrapmm("__shl")
    wrapmm("__shr")
    wrapmm("__add")
    wrapmm("__concat")
    wrapmm("__len")
    wrapmm("__eq")
    wrapmm("__lt")
    wrapmm("__le")
    wrapmm("__newindex")
    wrapmm("__call")
    wrapmm("__name")
    wrapmm("__close")
                
    return ud
end

-- Helper function
function accessOnMain(obj)
    return wrap(true, obj)
end

-- Override pairs & ipairs to ensure objects
-- with access forced to the main thread are
-- fully populated before iterating
local _pairs = pairs
function pairs(t)
    local mt = getmetatable(t)
    if mt and rawget(mt, "___lazyObj") then
        wrap(false, t)
    end
    return _pairs(t)
end

local _ipairs = ipairs
function ipairs(t)
    local mt = getmetatable(t)
    if mt and rawget(mt, "___lazyObj") then
        wrap(false, t)
    end
    return _ipairs(t)
end

-- Thread compatibility
function useThreadCompatLayer()
    layout = accessOnMain(layout)
    viewer = accessOnMain(viewer)
    asset = accessOnMain(asset)
end

-- Use the compat layer by default
useThreadCompatLayer()
