--- @class Promise
-- JavaScript like promises
Promise = class()
local Chain = class()

Promise.__name = "Promise"
--- @function Promise (func)
-- Promise constructor.
-- @param func Function to execute on a new thread. This function should accept 2 parameters, resolve & reject.
-- @return New Promise object
function Promise:init(func)
    
    -- No results
    self.val = nil
    self.err = nil
    
    self.sem = Semaphore()
    
    -- Chained functions
    self.chains = {}
    
    -- Run the provided function on a separate thread
    if func then
        Thread(function()
            func(function(...)
                self:resolve(...)
            end,
            function(err)
                self:reject(err)
            end)
        end)
    end
end

--- @function Promise:resolve (...)
-- Resolve the promise with the provided values.
-- @param ... values to resolve the promise with. These will be passed to any chained promises.
function Promise:resolve(...)
    self.val = {...}
        
    -- Pass along promise chain
    for _,chain in ipairs(self.chains) do
        chain:fulfill(self.val)
    end
        
    -- Resume threads that are waiting on us
    self.sem:signalAll()
end

--- @function Promise:reject (err)
-- Reject the promise with the provided error.
-- @param err Error to reject the promise with. This will be passed to any chained promises.
function Promise:reject(err)
    self.err = err
    
    -- Pass along promise chain
    for _,chain in ipairs(self.chains) do
        chain:reject(err)
    end
    
    -- Resume threads that are waiting on us
    self.sem:signalAll()
end

--- @function Promise:next (onFullfilled, onRejected)
-- Chain handler functions to be called when the promise is resolved.
-- The next() method returns a Promise. It takes up to two arguments: callback functions for the success and failure cases of the Promise.
-- @param onFullfilled (optional) A Function called if the Promise is fulfilled. This function can have multiple arguments, the fulfillment values. If it is not a function, it is internally replaced with an "Identity" function (it returns the received argument).
-- @param onRejected (optional) A Function called if the Promise is rejected. This function has one argument, the rejection reason. If it is not a function, it is internally replaced with a "Thrower" function (it throws an error it received as argument).
-- @return A new promise to be resolved with the result of onFullfilled or onRejected.
function Promise:next(onFulfilled, onRejected)
    local chain = Chain(onFulfilled, onRejected)
    table.insert(self.chains, chain)
    
    if self.val then
        chain:fulfill(self.val)
    elseif self.err then
        chain:reject(self.err)
    end
    
    return chain.promise
end

--- @function Promise:catch (onRejected)
function Promise:catch(onRejected)
    return self:next(nil, onRejected)
end

--- @function Promise:finally (onFinally)
function Promise:finally(onFinally)
    local fn = function()
        return onFinally()
    end
    return self:next(fn, fn)
end

--- @function Promise.all (promises)
function Promise.all(promises)
    local pr = Promise()
    
    local count = #promises
    local res = {}
    
    local function onRejected(err)
        pr:reject(err)
    end
    
    for i,p in ipairs(promises) do
        p:next(function(...)
            -- If not rejected already
            if pr.err == nil then
                res[i] = {...} -- add the result
                
                -- Resolve the 'all' promise if
                -- all promises have resolved.
                count = count - 1
                if count == 0 then
                    pr:resolve(res)
                end
            end
        end, onRejected)
    end
    
    return pr
end

--- @function Promise:await (fast)
function Promise:await(fast)
    if self.val == nil and self.err == nil then
        self.sem:wait(fast)
    end
    if self.err then error(self.err:gsub("^.-:[0-9]+: ", ""), 2) end
    if self.val then return table.unpack(self.val) end
end

local function identity(arg)
    return arg
end

local function thrower(err)
    error(err:gsub("Async:[0-9]+: ", "")) -- Strip error line
end

function Chain:init(onFulfilled, onRejected)
    self.promise = Promise()
    self.onFullfilled = onFulfilled or identity
    self.onRejected = onRejected or thrower
end

function Chain:fulfill(val)
    Thread(function()
        local r = table.pack(pcall(self.onFullfilled, table.unpack(val)))
        if r[1] then
            self.promise:resolve(select(2, table.unpack(r)))
        else
            self.promise:reject(r[2])
        end
    end)
end

function Chain:reject(err)
    Thread(function()
        local r = table.pack(pcall(self.onRejected, err))
        if r[1] then
            self.promise:resolve(select(2, table.unpack(r)))
        else
            self.promise:reject(r[2])
        end
    end)
end

--- @function async (func)
-- Returns a wrapped function that when called will return a promise.
--
-- The called function will be executed asynchronously on a new thread and the returned promise will be resolved with the result of the function call.
--
-- This allows any function to be turned into an asynchronous function with little effort.
-- @param func The function to be wrapped.
-- @return The wrapped function
function async(func)
    return function(...)
        local params = {...}
        return Promise(function(resolve, reject)
            local res = table.pack(pcall(func, table.unpack(params)))
            if res[1] == true then
                -- Resolve & return all results
                resolve(select(2, table.unpack(res)))
            else
                -- Reject and return error message
                reject(res[2])
            end
        end)
    end
end

--- @function await (promise, fast)
-- Wait for the given promise to resolve.
-- @param promise The promise to wait for.
-- @param fast (default false)
-- When false, execution will not resume on this thread until the next frame at the earliest.
-- When true, execution can return to this thread in the current frame.
-- @return The values the promise was resolved with.
function await(promise, fast)
    return promise:await(fast)
end
