-- Various functions overridden to run on the main thread.
-- If this is not done, most of these functions will simply
-- not work at all from a Thread.

--- @module http
-- Modifications to Codea's built-in http.request function.

--- @function http.request (url, success, fail, parameters)
-- Identical to the built-in function but callbacks will be executed on a thread to allow for debugging within the callbacks.
-- @param url URL to request
-- @param success Callback to be executed in the case of a successful request.
-- @param fail Callback to be executed in the case of a failed request.
-- @param parameters Request parameters.
local http_request = http.request
http.request = function(url, success, fail, parameters)
    Thread.runOnMain(http_request, url, Thread.callback(success, true), Thread.callback(fail, true), parameters)
end

--- @function http.requestProm (url, parameters)
-- Identical to the built-in function but returns results using a Promise object.
-- @param url URL to request
-- @param parameters Request parameters.
-- @return Promise object that will be resolved or rejected with the results of the request.
function http.requestProm(url, parameters)
    return Promise(function(resolve, reject)
        http.request(url, resolve, reject, parameters)
    end)
end

--- @function http.requestSync (url, parameters)
-- Identical to the built-in function but blocks waiting until the request completes (success or failure).
-- @param url URL to request
-- @param parameters Request parameters.
-- @return In the case of a successful request, a table in the form { response, statusCode, headers }
-- In the case of a failed request, a table in the form { nil, errorMessage }
function http.requestSync(url, parameters)
    
    local result = nil
    local sem = Semaphore()
    
    Thread.runOnMain(http_request, url,
        function(response, status, headers) -- Success
            result = { response, status, headers }
            sem:signal()
        end,
        function(err) -- Fail
            result = { nil, err }
            sem:signal()
        end, parameters)
    
    sem:wait()
    
    return table.unpack(result)
end

local terminators = {}
function registerTerminator(func)
    table.insert(terminators, func)
end
function willClose()
    for _,term in ipairs(terminators) do
        term()
    end
end
