local socket = require('socket')
local coroutine = coroutine

-- Thread management values
local threads = {}
local current_thread = nil
local sbFlushImage = image(1,1)
local async_queue = {}

local kNoParams = {}

-- Thread management task
local managementTask = tween.interval(0.001, function()
    
    -- Reset per frame flags
    for _,thread in ipairs(threads) do
        thread.done_this_frame = false
    end
    
    -- Keep running while threads are still executing
    local thread_ran = true
    while thread_ran do
        
        -- Reset flag
        thread_ran = false
        
        local deadThreads = {}
    
        for i,thread in ipairs(threads) do
            
            local dbg_thread = Debugger.thread
            
            -- Is the coroutine dead?
            if coroutine.status(thread.co) == "dead" then
                thread.dead = true -- set flag
                table.insert(deadThreads, i)
                
            -- Has it explicitly requested a detach?
            elseif thread.requestDetach then
                thread.requestDetach = false -- cancel the request, it'll be handled
                table.insert(deadThreads, i)
                
            -- Has the thread finished for this frame?
            elseif thread.done_this_frame then
                
            -- Ignore this thread if it's sleeping
            elseif thread.sleep_end and socket.gettime() < thread.sleep_end then
                
                -- If the sleeping thread is 'blocking' pretend we ran
                if thread.blocking then
                    thread_ran = true
                end
                
            -- If we're debugging, only run the debugged thread
            elseif dbg_thread ~= nil and thread ~= dbg_thread then
                
            -- All checks have passed. Resume the thread!
            else
                -- Track the current thread
                current_thread = thread
                thread_ran = true
                
                while true do
                    local res = table.pack(coroutine.resume(thread.co, table.unpack(thread.params or kNoParams)))
                    thread.params = {} -- clear the params
                    if not res[1] and thread.detachOnError then
                        thread.err = string.format("%s\n\n%s", res[2], debug.traceback(thread.co))
                        thread.dead = true -- set flag
                        table.insert(deadThreads, i)
                        break
                    elseif not res[1] then
                        -- Propagate the error
                        error(string.format("\n%s\n\n%s\n\nIGNORE TRACE BELOW:", res[2], debug.traceback(thread.co)))
                    elseif res[2] then
                        -- Call the value returned from the resume
                        -- and resume again.
                        -- This is how a thread forces something to
                        -- run on the main thread.
                        current_thread = nil
                        thread.params = table.pack(res[2](select(3, table.unpack(res))))
                        current_thread = thread
                    else
                        -- Pause execution
                        break
                    end
                end
                
                current_thread = nil
            end
        end
        
        -- Remove dead threads
        for i=#deadThreads,1,-1 do
            table.remove(threads, deadThreads[i])
        end
    end
        
    -- Render an offscreen sprite to flush the
    -- spritebatcher.
    -- This ensures sprite rendering done on
    -- threads is actually visible on screen.
    -- A wild guess is that the batcher is only
    -- flushed after the draw() call, and not
    -- after tween handling.
    spriteMode(CORNER)
    sprite(sbFlushImage, -1, -1)
end)
managementTask.noAsyncCallback = true
managementTask.ignoreDebug = true


--- @class Thread
-- Cooperative thread object based on Lua coroutines.
Thread = class()
Thread.__name = "Codea+ Thread"

--- @function Thread (func)
-- Thread constructor.
-- @param func Function to execute on the new thread.
-- @usage function setup()
--     -- Start a thread that will print its value once per frame
--     Thread(function()
--
--         -- Our thread local counter
--         local i = 0
--
--         -- Loop forever
--         while true do
--
--             -- Print and increment counter
--             print("Thread counter", i)
--             i = i + 1
--             
--             -- Yield the thread to stop execution for the current frame
--             yield()
--         end
--     end)
-- end
function Thread:init(fn, name)
    self.co = coroutine.create(fn)
    self.blocking = false
    self.name = name
    self.detachOnError = false
    
    table.insert(threads, self)
end

--- @function Thread.runOnMain (func, ...)
-- Immediately executes the given function on the main thread.
--
-- Various Codea functions do not function correctly when used from a coroutine so use this to temporarily hand execution back to the main thread.
-- @param func Function to run on the main thread.
-- @param ... Parameters to supply to the provided function.
-- @usage -- Takes a screenshot from the main thread.
-- Thread.runOnMain(function()
--     screenshot = viewer.snapshot()
-- end)
-- @usage -- Prints a message from the main thread.
-- Thread.runOnMain(print, "Hello main thread!")
function Thread.runOnMain(func, ...)
    if coroutine.isyieldable() then
        return coroutine.yield(func, ...)
    end
    return func(...)
end

function Thread.runAsync(func, ...)
    if func == nil or Debugger.thread then return end
    local params = {...}
    for i,p in ipairs(params) do
        -- Convert userdata param to a table
        if type(p) == "userdata" and p.___getters then
            local np = {}
            for k,fn in pairs(p.___getters) do
                np[k] = fn(p)
            end
            params[i] = np
        end
    end
    table.insert(async_queue, { 
        fn = func,
        params = params
    })
end

--- @function Thread.callback (func, ignoreDebugger)
-- Wraps a function to ensure it is executed on a thread.
--
-- By default wrapped functions will not execute (or be queued) if the debugger is active. Set ignoreDebugger to true if the callback should be executed regardless of the debugger.
-- @param func The function to be wrapped.
-- @param ignoreDebugger If true then the function will execute regardless of the state of the debugger.
-- @return The wrapped function.
-- @usage local function success(data, statusCode, headers)
--     print(statusCode)
-- end
--
-- -- If this weren't wrapped, we couldn't call
-- -- dbg() from inside the callback
-- local failure = Thread.callback(function(err)
--     print(err)
--     dbg()
-- end)
--
-- Thread.runOnMain(
--     http.request,
--     "https://codeawebrepo.co.uk/manifest.json",
--     success, failure
-- )
function Thread.callback(fn, ignoreDebugger)
    return function(...)
        if fn == nil or (Debugger.thread and not ignoreDebugger) then return end
        table.insert(async_queue, { 
            fn = fn,
            params = { ... }
        })
    end
end

--- @function Thread.wrapMain (func)
-- Wraps a function to ensure it is executed on the main thread.
-- @param func The function to wrap.
-- @return The wrapped function
function Thread.wrapMain(fn)
    return function(...)
        return Thread.runOnMain(fn, ...)
    end
end

--- @function Thread.current ()
-- Get the current thread.
-- @return The currently running thread, or nil if the main thread is running.
function Thread.current()
    return current_thread
end

--- @function Thread.allThreads ()
-- Get all active threads.
-- @return A table containing all active threads.
function Thread.allThreads()
    return threads
end

--- @function Thread.sleep (duration)
-- Put the current thread to sleep for a period of time.
-- @param duration Duration of sleep in seconds
function Thread.sleep(duration)
    current_thread.sleep_end = socket.gettime() + duration
    yield()
end


function Thread._wrapCoroutine(cr)
    local self = setmetatable({
        co = cr,
        blocking = false,
        detachOnError = true
    }, Thread)
    return self
end

--- @function Thread:detach ()
-- Remove a thread from the thread dispatch loop.
--
-- This eliminates all overhead for an idle thread. If you wish to resume execution on the thread, ensure you call attach().
function Thread:detach()
    self.requestDetach = true
end

--- @function Thread:attach ()
-- Add a previously detached thread to the thread dispatch loop.
function Thread:attach()
    if self.requestDetach == true then
        -- We're still actually attached so just cancel the
        -- detach request.
        self.requestDetach = false
    else
        -- Attach to the thread list
        table.insert(threads, self)
    end
end

--- @function yield (fast)
-- Yield execution on the current thread.
-- This allows cooperative multitasking during a frame & across frames.
-- @param fast (default false)
-- When false, execution will not resume on this thread until the next frame at the earliest.
-- When true, execution can return to this thread in the current frame.
function yield(fast)
    if not coroutine.isyieldable() then
        error("Unable to yield!")
    end
    current_thread.done_this_frame = (fast == nil or fast == false)
    coroutine.yield()
end

--- @function abort ()
-- Halts all threads and stops execution
function abort()
    objc.warning("abort.")
    threads = {}
    yield()
end

__wrapGlobal("touched", Thread.runAsync)
__wrapGlobal("hover", Thread.runAsync)
__wrapGlobal("scroll", Thread.runAsync)
__wrapGlobal("pinch", Thread.runAsync)
__wrapGlobal("keyboard", Thread.runAsync)
-- __wrapGlobal("willClose", Thread.runAsync) -- Don't wrap this

-- Async callback thread
Thread(function()
    
    -- We're a blocking thread as callbacks should
    -- be handled in the same frame as they were issued
    current_thread.blocking = true
    
    while true do
        for _,cb in ipairs(async_queue) do
            cb.fn(table.unpack(cb.params))
        end
        async_queue = {}
        
        yield()
    end
    
end)

-- These should do nothing
__wrapGlobal("setup", function() end)
__wrapGlobal("draw", function() end)

-- Thread to call setup() & draw()
Thread(function()
    
    -- We're a blocking thread
    current_thread.blocking = true
    
    -- Call setup() once
    local fn = __getWrappedGlobal("setup")
    if fn then fn() end
    
    -- Call draw() repeatedly
    while true do
        local fn = __getWrappedGlobal("draw")
        if fn then fn() end
        yield()
    end
end)