--- @module Debugger
    
Debugger = {
    thread = nil,
    debug_threads = false,
}

local saved_values = {}
local saved_viewer_mode = nil

local info = {
    traceback = {},
    locals = {},
    upvalues = {}
}

local function gatherDebugInfo()
    
    local traceback = info.traceback
    local locals = info.locals
    local upvalues = info.upvalues
    
    local level = 1
    while true do
        local info = debug.getinfo(Debugger.thread.co, level + 2, "Sl")
        if not info then break end
        
        locals[level] = {}
        upvalues[level] = {}
        
        if info.what == "C" then   -- is a C function?
            table.insert(traceback, string.format("[%d]C function", level))
            
        else   -- a Lua function
            local short_src = info.short_src:gsub('%[string "(.-)"%]', "%1")
            table.insert(traceback, string.format("[%d]%s:%d", level, short_src, info.currentline))
            
            -- Gather locals
            local i = 1
            while true do
                local name, val = debug.getlocal(Debugger.thread.co, level + 2, i)
                if not name then break end
                locals[level][name] = { val, i }
                i = i + 1
            end
            
            -- Gather upvalues
            local func = debug.getinfo(Debugger.thread.co, level + 2).func
            i = 1
            while true do
                local name, val = debug.getupvalue(func, i)
                if not name then break end
                upvalues[level][name] = { val, i }
                i = i + 1
            end
            
        end
        level = level + 1
    end
    
    info.traceback = table.concat(traceback, "\n")
end



local function continue()
    Debugger.thread = nil
    _G.c = saved_values.c
    _G.l = saved_values.l
    _G.sl = saved_values.sl
    _G.t = saved_values.t
    
    
    -- Clear traceback, locals & upvalues
    info.traceback = {}
    info.locals = {}
    info.upvalues = {}
end

-- Locals
local function locals(name, level)
    
    -- Default to display first stack frame
    name = name or 1
    
    if type(name) == "string" then
        if level == nil then
            
            for i=1,#info.locals do
                
                -- Check locals first
                local v = info.locals[i][name]
                if v == nil then
                    -- Then upvalues
                    v = info.upvalues[i][name]
                end
                
                if v then
                    v = v[1]
                    local typ = type(v)
                    if typ == "table" or typ == "function" then
                        print(name .. " = " .. tostring(v))
                    else
                        print(name .. " = " .. typ .. ": " .. tostring(v))
                    end
                    return v
                end
            end
            
            objc.warning("No local '" .. name .. "' in any stack frame")
        else
            if level > #info.locals then
                objc.warning("Invalid stack frame: " .. level)
                return
            end
            
            -- Check locals first
            local v = info.locals[level][name]
            if v == nil then
                -- Then upvalues
                v = info.upvalues[level][name]
            end
            
            if v then
                v = v[1]
                local typ = type(v)
                if typ == "table" or typ == "function" then
                    print(name .. " = " .. tostring(v))
                else
                    print(name .. " = " .. typ .. ": " .. tostring(v))
                end
                return v
            else
                objc.warning("No local '" .. name .. "' in stack frame " .. level)
            end
        end
            
    elseif type(name) == "number" then
        if name > #info.locals then
            objc.warning("Invalid stack frame: " .. name)
            return
        end
        
        local out = {"Locals (level " .. name .. "):"}
        for k,v in pairs(info.locals[name]) do
            
            v = v[1]
            local typ = type(v)
            
            if typ == "table" or typ == "function" then
                table.insert(out, k .. " = " .. tostring(v))
            else
                local vs = tostring(v)
                if string.len(vs) > 32 then
                    vs = "..."
                end
                
                table.insert(out, k .. " = " .. typ .. ": " .. vs)
            end
        end
        
        -- Append upvalues
        for k,v in pairs(info.upvalues[name]) do
            v = v[1]
            local typ = type(v)
            
            if typ == "table" or typ == "function" then
                table.insert(out, k .. " = " .. tostring(v))
            else
                local vs = tostring(v)
                if string.len(vs) > 32 then
                    vs = "..."
                end
                
                table.insert(out, k .. " = " .. typ .. ": " .. vs)
            end
        end
        
        print(table.concat(out, "\n"))
    else
        objc.warning("l: Invalid parameter type. Expects string or number.")
    end
end

local function setlocal(name, value, level)
    
    if level == nil then
        
        for i=1,#info.locals do
            -- Check locals first
            local v = info.locals[i][name]
            
            if v then
                -- Set the new value
                debug.setlocal(Debugger.thread.co, i + 2, v[2], value)
                v[1] = value
                print("Local '" .. name .. "' set in stack frame " .. i)
                return
            else
                -- Then upvalues
                v = info.upvalues[i][name]
                if v then
                    -- Set the new value
                    local func = debug.getinfo(Debugger.thread.co, i + 2).func
                    debug.setupvalue(func, v[2], value)
                    v[1] = value
                    print("Local '" .. name .. "' set in stack frame " .. i)
                    return
                end
            end
        end
            
        objc.warning("No local '" .. name .. "' in any stack frame")
    else
        if level > #info.locals then
            objc.warning("Invalid stack frame: " .. level)
            return
        end
        
        -- Check locals first
        local v = info.locals[level][name]
            
        if v then
            -- Set the new value
            debug.setlocal(Debugger.thread.co, level, v[2], value)
            v[1] = value
            print("Local '" .. name .. "' set in stack frame " .. level)
            return
        else
            -- Then upvalues
            v = info.upvalues[level][name]
            if v then
                -- Set the new value
                local func = debug.getinfo(Debugger.thread.co, level + 2).func
                debug.setupvalue(func, v[2], value)
                v[1] = value
                print("Local '" .. name .. "' set in stack frame " .. level)
                return
            end
        end
        
        objc.warning("No local '" .. name .. "' in stack frame " .. level)
    end
end

local function traceback()
    print("Trace:\n" .. info.traceback)
end

local function launch_debugger()
    Debugger.thread = Thread.current()
    saved_values.c = _G.c
    saved_values.l = _G.l
    saved_values.sl = _G.sl
    saved_values.t = _G.t
    _G.c = continue
    _G.l = locals
    _G.sl = setlocal
    _G.t = traceback
    
    -- Show sidepanel
    saved_viewer_mode = viewer.mode
    viewer.mode = OVERLAY
    
    -- Pause physics
    physics.pause()
end

--- @function dbg (condition)
-- Breaks into the debugger. If a condition is provided, the debugger is only triggered if the condition evaluates to true.
-- @param condition (optional) If true, the debugger will be triggered.
function dbg(condition)
    if condition == nil or condition == true then
        
        launch_debugger()
        gatherDebugInfo()
        
        -- Capture the current screen image
        local screen_image = viewer.snapshot()
        
        -- Print backtrace
        print("Debug break:\n" .. info.traceback)
        
        -- Yield until the debugger deactivates itself
        while Debugger.thread ~= nil do
            yield()
            
            -- Draw the current screen so we don't flicker
            pushStyle()
            spriteMode(CORNER)
            sprite(screen_image, 0, 0, WIDTH, HEIGHT)
            popStyle()
        end
        
        -- Restore sidepanel mode
        viewer.mode = saved_viewer_mode
    
        -- Resume physics
        physics.resume()
    end
end


--- @function c ()
-- Exit the debugger and continue execution.
--
-- <b>Important:</b> This is only to be called manually using the Codea sidepanel while the debugger is active.

--- @function l (nameOrLevel, level)
-- View local variables.
--
-- When called with no arguments this prints all local variables at the stack frame of the active 'dbg()' call.
--
-- Note: When listed, string values are omitted. Use l("&lt;variable_name&gt;", [stack_level]) to view the full value.
--
-- <b>Important:</b> This is only to be called manually using the Codea sidepanel while the debugger is active.
-- @param nameOrLevel (optional) Name of the variable to read <b>OR</b> stack level to list.
-- @param level (optional) Level to search for the given local variable name.
-- @usage -- Prints all local variables in the current stack frame
-- l()
-- @usage -- Prints all local variables in the next stack frame up (1 == current frame)
-- l(2)
-- @usage -- Prints the value of the local variable 'i'
-- -- in the first stack frame it is found.
-- l("i")
-- @usage -- Prints the value of the local variable 'i' in the 3rd stack frame.
-- l("i", 3)

--- @function sl (name, value, level)
-- Set a local variable.
--
-- <b>Important:</b> This is only to be called manually using the Codea sidepanel while the debugger is active.

--- @function t ()
-- Print the backtrace again.
--
-- <b>Important:</b> This is only to be called manually using the Codea sidepanel while the debugger is active.
 