local re = require('re')

local grammar = table.concat({
    "chunk <- {| block |}",
    
    "block <- ((comment / (define / (undef / ifend))) / {| {:tag: ''->'code':} code |})*",
    
    -- Preprocessor 'defines'
    "define <- {| {:tag: ''->'define' :} 'DEF(' space {:id: id :} space ',' space {:val: (string / 'true' / 'false' / 'nil') :} ')' |} space",
    
    -- Remove preprocessor 'defines'
    "undef <- {| {:tag: ''->'undef' :} 'UNDEF(' space {:id: id :} space ')' |}",
    
    -- Preprocessor conditional compilation
    "ifend <- {| {:tag: ''->'ifend' :} 'IF(' space {:cond: id :} space ')' {:block: {|block|} :} 'END()' |}",
    
    "code <- {:code: (!define !ifend !'END()' .)+ :}",
    
    "comment <- {| {:tag: ''->'comment' :} (('--' {:val: longstring :}) / ('--' {:val: [^\n]* :})) |}", 
    
    [[string <- ((('"' {(('\\' .) / (!'"' .))*} '"')
        / ("'" {(('\\' .) / (!"'" .))*} "'"))
        / longstring)]],
    "longstring <- '[' {:init: '='* :} '[' {stringblock} ']' =init ']'",
    "stringblock <- &(']' =init ']') / . stringblock",
    
    "id <- [a-zA-Z_][a-zA-Z0-9_]*",
    "space <- %s*" -- Consume all whitespace
}, '\n')
local pattern = re.compile(grammar)
    
-- Shared preprocessor definitions between files
local defines = {}

local PREPROCESSED = false
local function _preprocess(srcAsset, printProcessed)
    
    -- If we've already been preprocessed, return false so the source can load fully
    if PREPROCESSED then
        return false
    end
    
    -- Read original source
    local f = io.open(srcAsset.path, "rb")
    local src = f:read("*a")
    f:close()
    
    -- Preprocess the source
    local ast = re.match(src, pattern)
    
    --print(json.encode(ast, {indent=true}))

    local src = {}
    
    local function evalDefine(node)
        local val = node.val
        if val == "true" then
            node.val = true
        elseif val == "false" then
            node.val = false
        end
    end
    
    local function insertCode(code)
        local loop, matches = true, 0
        while loop do
            loop = false
            for k,v in pairs(defines) do
                code, matches = code:gsub("([^a-zA-Z0-9_])" .. k .. "([^a-zA-Z0-9_])", "%1" .. tostring(v) .. "%2")
                
                -- Keep looping until we stop replacing preprocessor defines
                if matches > 0 then loop = true end
            end
        end
        table.insert(src, code)
    end
    
    local handlers = nil
    local function processAst(ast)
        for _,node in ipairs(ast) do
            handlers[node.tag](node)
        end
    end
    
    handlers = {
        code = function(node)
            insertCode(node.code)
        end,
        define = function(node)
            evalDefine(node)
            defines[node.id] = node.val
        end,
        undef = function(node)
            defines[node.id] = nil
        end,
        ifend = function(node)
            if defines[node.cond] then
                processAst(node.block)
            end
        end,
        comment = function() end
    }

    processAst(ast)
    
    src = table.concat(src)
    if printProcessed == true then
        print(src)
    end
    
    -- Set flag so we don't preprocess again when we load
    PREPROCESSED = true
    
    -- Load the modified source
    local chunkname = srcAsset.name
    chunkname = chunkname:match("(.-)%.lua") or chunkname
    local fn, err = load(src, chunkname)
    if fn then
        fn()
    else
        error(err, 2)
    end
    
    -- Unset flag so other files can be preprocessed
    PREPROCESSED = false
    return true
end

--- @module Preprocessor
-- Lua preprocessor allowing for basic conditional compilation and text substitution at compile time.
--
-- This allows for code segments to be entirely omitted from the executed code which can be incredibly useful when you want your code to be performant AND debugable.

--- @function Preprocess (srcAsset, print)
-- Preprocesses and loads the given source asset.
-- @param srcAsset The Codea asset key corresponding to the file to be loaded.
-- @param print If true, the final processed source code will be printed to the console.
-- @usage -- Preprocessing should always be done at the top
-- -- of your source file in an if statement like this.
-- -- If this is not done inside an if statement this will not work.
-- if Preprocess(asset.Main) then return end
--
-- -- Preprocessor defines
-- DEF(DEBUG, true)
--
-- -- Strings must be within a string themselves
-- -- in order to be inserted correctly into the source code.
-- DEF(kWelcomeMessage, "'Welcome to the preprocessor!'")
-- 
-- function setup()
--     print(kWelcomeMessage)
--     IF(DEBUG)
--         -- Only include this print statement when the 'DEBUG' preprocessor
--         -- value is true.
--         print("Debug message")
--     END()
-- end
function Preprocess(srcAsset, printProcessed)
    local success, r = pcall(_preprocess, srcAsset, printProcessed)
    if success then return r end
    --error(r, 2)
    print(r)
end

--- @function DEF (name, value)
-- Adds a preprocessor defined value.
-- A value set using this function will not be a conventional variable, rather at compile time instances of the key in the source code will be replaced with the value provided to this function.
-- @param id The name given to the provided value.
-- @param value The value to replace occurances with. This can be a string, true, false or nil.
-- @usage DEF(kWelcomeMessage, '"Hello Codea"')
--
-- print(kWelcomeMessage) -- After preprocessing becomes 'print("Hello Codea")'
-- @usage DEF(kExperimentalFeatures, false)
--
-- ...
--
-- IF(kExperimentalFeatures)
--     -- Do unstable experimental thing. Funny that!
--     ...
-- END()
function DEF()
    error("DEF() Called! This should not happen so check your preprocessor syntax!")
end

--- @function UNDEF (id)
-- Undefines a value previously defined with DEF(id).
-- @param id The name of the value to be undefined.
function UNDEF()
    error("UNDEF() Called! This should not happen so check your preprocessor syntax!")
end

--- @function IF (id)
-- Start a conditional compilation block.
function IF()
    error("IF() Called! This should not happen so check your preprocessor syntax!")
end

--- @function END ()
-- End a conditional compilation block.
function END()
    error("END() Called! This should not happen so check your preprocessor syntax!")
end
