-- Various functions overridden to run on the main thread.
-- If this is not done, most of these functions will simply
-- not work at all from a Thread.

--- @module http
-- Modifications to Codea's built-in http.request function.

--- @function http.request (url, success, fail, parameters)
-- Identical to the built-in function but callbacks will be executed on a thread to allow for debugging within the callbacks.
-- @param url URL to request
-- @param success Callback to be executed in the case of a successful request.
-- @param fail Callback to be executed in the case of a failed request.
-- @param parameters Request parameters.
local http_request = http.request
http.request = function(url, success, fail, parameters)
    Thread.runOnMain(http_request, url, Thread.callback(success, true), Thread.callback(fail, true), parameters)
end

--- @function http.requestProm (url, parameters)
-- Identical to the built-in function but returns results using a Promise object.
-- @param url URL to request
-- @param parameters Request parameters.
-- @return Promise object that will be resolved or rejected with the results of the request.
function http.requestProm(url, parameters)
    return Promise(function(resolve, reject)
        http.request(url, resolve, reject, parameters)
    end)
end

--- @function http.requestSync (url, parameters)
-- Identical to the built-in function but blocks waiting until the request completes (success or failure).
-- @param url URL to request
-- @param parameters Request parameters.
-- @return In the case of a successful request, a table in the form { response, statusCode, headers }
-- In the case of a failed request, a table in the form { nil, errorMessage }
function http.requestSync(url, parameters)
    
    local result = nil
    local sem = Semaphore()
    
    Thread.runOnMain(http_request, url,
        function(response, status, headers) -- Success
            result = { response, status, headers }
            sem:signal()
        end,
        function(err) -- Fail
            result = { nil, err }
            sem:signal()
        end, parameters)
    
    sem:wait()
    
    return table.unpack(result)
end

--- @module Util
-- Various useful functions & utilities.

--- @function registerTerminator (func)
-- Registers a function to be called when the project is stopped.
-- This is akin to Codea's 'willClose()' function but allows multiple callback functions to be registered.
-- @param func The function to be registered as a terminator.

--- @function addTerminator (func)
-- Alias of registerTerminator().

--- @function unregisterTerminator (func)
-- Removes a previously registered function from the list of terminator functions.
-- @param func The function to be unregistered.

--- @function removeTerminator (func)
-- Alias of unregisterTerminator().
local terminators = {}
function registerTerminator(func)
    terminators[func] = func
end
function unregisterTerminator(func)
    terminators[func] = nil
end
addTerminator = registerTerminator
removeTerminator = unregisterTerminator
__wrapGlobal("willClose", function(fn)
    for _,term in pairs(terminators) do
        term() -- call registered terminator
    end
    fn() -- call original
end)
-- blank willClose to initialise the wrapper above
function willClose() end

--- @function enableLogging (silent, logfile)
-- Enables logging all 'print' output to the project tab with the given name.
--
-- This also prefixes all output with the current time.
-- @param silent If true, nothing will be printed to the sidepanel, only the logfile.
-- @param logfile Name of the project tab to write our log to.
local _print = print
function enableLogging(silent, logfile)
    logfile = logfile or "output"
    saveProjectTab(logfile, "")
    local f = io.open(asset.path .. "/" .. logfile .. ".lua", "w")
    f:write("--[==========[\n]==========]")
    f:flush()
    print = function(...)
        local time = os.date("%X")
        if silent ~= true then _print(time, ...) end
        f:seek("cur", -12)
        f:write(time)
        for _,v in ipairs({...}) do
            f:write('\t' .. tostring(v))
        end
        f:write('\n]==========]')
        f:flush()
    end
end
    