----------------- DO NOT REMOVE OR MOVE -----------------
-- Ensure Codea doesn't load this file automatically
-- This MUST be at the top of this file!
if WRL and not WRL.loading then return end
--------------- END DO NOT REMOVE OR MOVE ---------------
-- Contents:
--    Util.lua
--    CrashReporter.lua
--    TaskLoop.lua
--    Environments.lua
--    Lib.lua
--    Zip.lua
--    LoadGuard.lua
--    WRL.lua

------------------------------
-- Util.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

-- Create WRL global
WRL = {
    ProjectName = asset.name:gsub(".codea", "")
}
end
------------------------------
-- CrashReporter.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

-- Avoid coroutine table overrides
local coroutine = coroutine

WRL.Crash = {}

local function stripDocPrefix(str)
    return str:gsub(".-/Documents/", "")
end

WRL.CrashReport = class()

function WRL.CrashReport:init(errMsg)
    self.report = {
        project = WRL.ProjectName,
        date = os.date("!%Y-%m-%d %T") .. " (UTC)",
        error = (errMsg and stripDocPrefix(errMsg)) or "nil"
    }
    
    local version = "Local"
    do
        -- Get the version of the running project
        local verAsset = asset["webrepo_version"]
        if verAsset then
            local f = io.open(verAsset.path, "r")
            version = f:read("*a")
            f:close()
        end
        self.report.version = version
        
        --[[
        if verAsset then
            -- Compare project checksum to detect modifications
            addToReport("Modified: <unimplemented>")
        end
        ]]
    end
end

function WRL.CrashReport:dumpCoroutine(co, name, crashed, offset)
    self.report.coroutines = self.report.coroutines or {}
    
    -- Add our new coroutine dump
    local dump = {}
    table.insert(self.report.coroutines, dump)
    
    dump.name = name
    dump.status = coroutine.status(co)
    dump.crashed = crashed or nil
    dump.callstack = {}

    local i = offset or 0
    while true do
        local info = debug.getinfo(co, i)
        if info == nil then break end
        
        -- Create a new stack frame
        local frame = {}
        table.insert(dump.callstack, frame)
            
        -- Add thread file:line
        frame.location = {
            file = stripDocPrefix(info.short_src),
            line = info.currentline,
            func = info.name or "?"
        }
            
        -- Add locals to report
        if debug.getlocal(co, i, 1) then
            frame.locals = {}
            local j = 1
            while true do
                local k, v = debug.getlocal(co, i, j)
                if not k then break end
                local v = tostring(v)
                if string.len(v) > 64 then
                    v = v:sub(1, 64) .. "..."
                end
                if v:match("[^%g%s]") then
                    v = "<binary string>"
                end
                frame.locals[k] = v
                j = j + 1
            end
        end
            
        -- Add upvalues to report
        if info.nups > 0 then
            frame.upvalues = {}
            local j = 1
            while true do
                local k, v = debug.getupvalue(info.func, j)
                if not k then break end
                local v = tostring(v)
                if string.len(v) > 64 then
                    v = v:sub(1, 64) .. "..."
                end
                if v:match("[^%g%s]") then
                    v = "<binary string>"
                end
                frame.upvalues[k] = v
                j = j + 1
            end
        end
            
        i = i + 1
    end
end

function WRL.CrashReport:send()
    local jsonState = {
        indent = true,
        level = 1,
        keyorder = {
            "project",
            "date",
            "version",
            "error",
            "coroutines"
        }
    }

    -- Stop drawing
    _G.draw = nil
    
    local function initParameters(clipboardOnly)
        parameter.clear()
        if not clipboardOnly then
            parameter.text("Notes")
            parameter.text("Reporter", "anonymous")
            parameter.action("Send crash report", function()
                -- Create the payload
                if #Notes == 0 then Notes = "N/A" end
                Notes = Notes:gsub("…", "...") -- Replace weird ellipses char...
                
                self.report.notes = Notes
                self.report.reporter = Reporter
                    
                --uploadToBayfiles(json.encode(payload), "crash_report.txt", function(url)
                    -- Link to the uploaded report
                    --payload.report = url
                        
                    -- Send the crash report
                    http.request("https://codeawebrepo.co.uk/api/crash", function(_, status)
                        if status == 200 then
                            print("Report sent! Thank you for reporting your issue.")
                            initParameters(true)
                        else
                            print("Failed to send. Please try again.")
                        end
                    end, function()
                        print("Failed to send. Please try again.")
                    end, {
                        method = "POST",
                        headers = {
                            ["Content-Type"] = "application/json"
                        },
                        data = json.encode(self.report, jsonState)
                    })
                --end)
                    
                print("Sending. Please wait...")
            end)
        end
        parameter.action("Copy report to clipboard", function()
            pasteboard.copy("--[==[\n" .. json.encode(self.report, jsonState) .. "\n]==]")
            print("Copied to clipboard")
        end)
    end
        
    viewer.mode = OVERLAY
    initParameters()
        
    --output.clear()
    print("If you would like to upload your crash report please press the button above.\n\nPlease provide any useful info you may have (what you were doing at the time, any modifications etc.) and your forum name if you wish.")
    
    
    local onScreenMessage = "Crashed!\n\t" .. self.report.error
    _G.tween.update = function()
        background(132, 30, 30)
            
        -- Setup text
        textWrapWidth(WIDTH-20)
        textMode(CENTER)
        textAlign(LEFT)
        fontSize(24)
        fill(255)
        font("Futura-MediumItalic")
            
        -- Draw report
        text(onScreenMessage, WIDTH/2, HEIGHT/2)
    end
end

function WRL.CrashHandler(msg)
    local report = WRL.CrashReport(msg)
    report:dumpCoroutine(coroutine.running(), nil, true, 3)
    report:send()
    return msg
end

function WRL.CrashableCall(fn, ...)
    return xpcall(fn, WRL.CrashHandler, ...)
end

end
------------------------------
-- TaskLoop.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

-- Tasks table
WRL.Tasks = {}

----------------------------------------
-- Task management
-- (Only used during project loading)
----------------------------------------
local tasks = {} -- Array of tasks
local taskLoop -- Loop function
local function nextTask()
    local task = table.remove(tasks)
    if task then
        local s, status = pcall(task.fn, table.unpack(task.args))
        if s then
            return status
        else
            local err = status
            if err and err:match("__TASK_LOOP_YIELD__") then
                taskLoop()
                return false
            else
                tasks = {} -- Don't run any more tasks
                WRL.CrashHandler(err)
            end
        end
    else
        return false
    end
end

-- Start task loop tween
taskLoop = function()
    tween.delay(0.00001, function()
        -- Keep executing tasks until we're told to stop
        while nextTask() do end
    end)
end
taskLoop()

WRL.Tasks.delay = function(frames)
    for i=1,frames do
        WRL.Tasks.addTask(function()
            taskLoop()
            return false
        end)
    end
    taskLoop()
    return false
end

-- Global functions
function WRL.Tasks.addTask(fn, ...)
    table.insert(tasks, { fn = fn, args = table.pack(...) })
end
function WRL.Tasks.taskLoopYield()
    error("__TASK_LOOP_YIELD__", 0)
end
end
------------------------------
-- Environments.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

----------------------------------------
-- Environment management routines
----------------------------------------

WRL.Env = {}

-- Local copy of ipairs
local _Gipairs = _G.ipairs

-- Creates an environment table for the project, including dependencies
function WRL.Env.createInternal(meta)
    
    -- Get the asset root of the project
    local assetRoot
    if meta.version == "__LOCAL__" then -- Local project
        assetRoot = asset.documents[meta.name .. ".codea"]
    else -- Remote project
        assetRoot = asset.documents.webrepolib_cache[meta.name][meta.version][meta.name .. ".codea"]
    end
    
    local aorig = {
        documents = asset.documents,
        builtin = asset.builtin
    }
    
    -- New asset root table
    local newAsset = setmetatable({}, {
        __index = function(t, k)
            return aorig[k] or assetRoot[k]
        end,
        __concat = function(t, v)
            return assetRoot .. v
        end,
        __tostring = function() return tostring(assetRoot) end
    })
    
    -- Our new environment
    local env = {
        asset = newAsset
    }
    
    local depenvs = {} -- Blank dependent envs table.
    -- Passthrough to _G for read access if depenvs doesn't contain the value
    setmetatable(env, {
        __index = function(t, k)
            for _,env in _Gipairs(depenvs) do
                local v = env[k]
                if v then return v end
            end
            return _G[k]
        end
    })
    
    -- Return the env & the depenvs table so we can include deps
    -- after the fact
    return env, depenvs
end

-- Copies internal env members to new table, ignoring the metatable
-- in use for internal envs.
function WRL.Env.createExternal(internalEnv)
    local envExtern = {}
    for k,v in pairs(internalEnv) do
        envExtern[k] = v
    end
    return envExtern
end

-- Copies members of one environment table to another.
function WRL.Env.import(from, to)
    to = to or _ENV
    for k,v in pairs(from) do
        to[k] = v
    end
end
end
------------------------------
-- Lib.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

-- Options:
--  version     - Specify the version of the dependency to load.
--  loadmain    - Loads the main file of the project/dependency
--  import      - Makes the env of the dependency available in the dependent project.
--  export      - Copies the env of the dependency to the dependent project's env.
--  deps        - Dependencies to be loaded prior to the current project.
--  devdeps     - Dependencies to be loaded when the project is launched from Codea
--  loaddev     - Forces devdeps to be used as dependencies.
--                  (default for the launching project)

--[[
@module: WebRepoLib
@desc: WebRepoLib makes accessing and using community libraries a breeze.
---<h3>Getting Started</h3>Everything revolves around the `WRL.lua` file/tab. When a project contains this file and adds WebRepoLib as a dependency, the magic starts.

`WRL.lua` describes your project's dependencies and how you intend to use them.
To add a dependency, define the `deps` table in `WRL.lua` and add your dependency to it:
```
deps = {
    "Codea+thread" -- Import from WebRepo.
}
```
When adding a new file to a project using WebRepoLib, when launching you will see a dependency error asking you to close and re-open your project. Please do as it asks, return to Codea's project browser and re-open the project.  WebRepoLib detects new files and adds guards to ensure they are loaded at the correct time. Only by re-opening your project will those changes be visible. <b>DO NOT</b> remove the load guards from projects where WebRepoLib is used.
These load guards look like this:
```
----------------- DO NOT REMOVE OR MOVE -----------------
-- Ensure Codea doesn't load this file automatically
-- This MUST be at the top of this file!
if WRL and not WRL.loading then return end
--------------- END DO NOT REMOVE OR MOVE ---------------
```
---<h3>Configuration</h3>Each included dependency can be configured in different ways.
This is done by adding a configuration table to a dependency:
```
deps = {
    { "Codea+thread", { ... } }
}
```

<b>Versions:</b>
By default, WebRepoLib will always use the latest version of any dependency available on WebRepo.
You can also fix the version you with to use by including a version in the config table:
```
deps = {
    { "Codea+thread", { version = "1.1.0" } }
}
```
<b>Local projects:</b>
By prepending your dependency's name with a '~' you can load dependencies from your own Codea folder on your device.
This is most useful during development with multiple dependent projects.
```
deps = {
    "~Codea+thread" -- Import local 'Codea+thread' project
}
```
<b>Access:</b>
By default, a dependency is only available to the project that has included it. This is done by making the dependency's environment accessible via the current project's environment's metatable. This allows projects to maintain their own environments without directly manipulating the environment of others.

This behaviour can be changed using the `export` flag:
```
deps = {
    { "~Codea+thread" { export = true } }
}
```When `export == true` the dependency's environment is copied into the including project. This allows dependencies to be made available further up the chain.

This is useful when a dependency defines callback functions used by Codea's runtime, e.g. setup(), draw() etc.

<b>loadmain:</b>
By default, only the project being launched by Codea will have its `Main.lua` file loaded.
By setting this flag in a dependency config, you are telling WebRepoLib that it should also load the dependency's `Main.lua` file as well.

WebRepo itself uses the same technique to ensure it runs correctly:
```
deps = {
    -- Always run the latest available WebRepo release.
    {"WebRepoImpl", {
        export = true, -- Export WebRepoImpl globals into our env.
        loadmain = true -- Load WebRepoImpl's 'Main' tab.
    }}
}
```
---<h3>Development dependencies</h3>Sometimes, you may need to include additional dependencies during development but not when your project is used as a dependency. To do this we use the `devdeps` table. Dependencies added to this table are only loaded when you are launching the project directly, and not as a dependency to another project.

The `devdeps` table is useful for things like docs and debuggers etc. and behaves exactly the same way as the `deps` table. This is how we include the codea+docs dependency used to display these docs:
```
devdeps = {
    { "Codea+docs" }
}
```
]]

-- For url escaping
local socket = require("socket")

-- Make a few functions locals
local addTask = WRL.Tasks.addTask
local taskLoopYield = WRL.Tasks.taskLoopYield
local createInternalEnv = WRL.Env.createInternal
local createExternalEnv = WRL.Env.createExternal
local importEnv = WRL.Env.import

-- print() with a prefix.
local _Gprint = print
local function print(...)
    _Gprint("[WRL]", ...)
end

----------------------------------------
-- Tasks
----------------------------------------
local TASK_kickoff
local TASK_validateProject
local TASK_installProject
local TASK_loadDependencies
local TASK_loadProject
local TASK_loadDeferredDependency
local TASK_go

local ENVS = {}
local manifest

TASK_kickoff = function()
    
    -- We're now doing our manual loading
    WRL.loading = true
    
    -- Current project
    local meta = {
        name = (asset.name):sub(1, -7),
        version = "__LOCAL__",
        loadmain = true, -- allows Main.lua to load,
        loaddev = true
    }
    
    -- Last task to be run should be us importing the
    -- current project's env & executing the setup() function.
    addTask(TASK_go, meta)
    
    -- Load the current project
    addTask(TASK_loadProject, meta)
    
    -- Get & load dependencies of the current project
    addTask(TASK_loadDependencies, meta)
    
    -- Setup file system & load WebRepo manifest
    local p = asset.documents.path .. "/webrepolib_cache"
    local fm = objc.NSFileManager.defaultManager
    if not fm:fileExistsAtPath_isDirectory_(p, true) then
        fm:createDirectoryAtPath_withIntermediateDirectories_attributes_error_(p, true, nil, nil)
    end
    
    -- Update the WebRepo manifest
    local function loadManifest(data)
        manifest = json.decode(data)
        for p, versions in pairs(manifest) do
            for _, v in ipairs(versions) do
                versions[v] = true
            end
        end
    end
    
    local manifestURL = "https://codeawebrepo.co.uk/manifest.json"
    local timedOut = false
    local function httpFail(err)
        -- Ignore if already loaded
        if manifest ~= nil then return end
        
        -- Read from disk instead?
        local data = readText(asset.documents.webrepolib_cache .. "manifest.json")
        if data == nil then
            error("Failed to GET " .. manifestURL .. " (" .. err .. ").")
        end
        print("Failed to update. Using cached manifest")
        loadManifest(data)
    end
    
    http.request(manifestURL, function(data, status)
        -- Ignore if already loaded
        if manifest ~= nil then return end
        
        if status ~= 200 then
            httpFail(tostring(status))
        else
            loadManifest(data)
            -- Save to disk
            saveText(asset.documents.webrepolib_cache .. "manifest.json", data)
        end
    end, httpFail)
    local reqTime = ElapsedTime
    
    -- Keep running the same task until the manifest is available
    print("Updating manifest")
    local function wait()
        
        -- Timeout?
        if (ElapsedTime - reqTime) > 2.0 and not timedOut then
            -- Read from disk if it's been downloaded already.
            -- If we don't already have it we'll continue waiting
            local data = readText(asset.documents.webrepolib_cache .. "manifest.json")
            if data then
                print("Timed out. Using cached manifest")
                loadManifest(data)
            end
            timedOut = true
        end
        
        -- Has it been downloaded?
        if manifest == nil then
            addTask(wait)
            taskLoopYield()
        end
        
        -- Success
        return true
    end
    addTask(wait)
    
    -- Success
    return true
end

TASK_validateProject = function(meta)
    if meta.name:match("^~") then
        meta.name = meta.name:sub(2) -- strip '~'
        meta.version = "__LOCAL__"
    else
        -- Get remote project
        
        -- Check project exists in manifest
        local versions = manifest[meta.name]
        if versions == nil then
            error("Project '" .. meta.name .. "' is not available.")
        end
        meta.version = meta.version or versions[#versions] -- Use latest by default
        if versions[meta.version] == nil then
            error("Project version '" .. meta.name .. "-" .. meta.version .. "' is not available.")
        end
        
        -- Check if the project is already installed
        if readText(asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/valid.txt") then
            -- Nothing to do!
            return true
        end
        
        -- Install the downloaded project
        addTask(TASK_installProject, meta)
    end
    
    -- Success
    return true
end

TASK_loadDependencies = function(meta)
    local wrl = {}
    
    if meta.version == "__LOCAL__" then
        -- Load the metadata file
        local fn = loadfile((asset.documents .. meta.name .. ".codea/WRL.lua").path, "t", wrl)
        if fn then
            if (WRL.CrashableCall(fn) == false) then
                return false
            end
        end
    else
        -- Remote project
        
        -- Load the metadata file
        local fn = loadfile((asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/" .. meta.name .. ".codea/WRL.lua").path, "t", wrl)
        if fn then
            if (WRL.CrashableCall(fn) == false) then
                return false
            end
        end
    end
    
    -- Allow overrides declared by the importer
    -- TODO: Use 'meta' tables to determine unique project loads.
    for k, v in pairs(wrl) do
        if meta[k] == nil then
            meta[k] = v
        end
    end
    
    local function convertDepFormat(deps)
        if not deps then return end
        for i,dep in ipairs(deps) do
            
            -- Allow string only deps (not in a table)
            if type(dep) == "string" then
                dep = { dep }
                deps[i] = dep
            end
            
            local extra = dep[2] or {}
            dep.name = dep[1]
            for k,v in pairs(dep[2] or {}) do
                dep[k] = v
            end
            dep[1] = nil
            dep[2] = nil
        end
    end
    
    -- Convert to internal meta format
    convertDepFormat(meta.deps)
    convertDepFormat(meta.devdeps)
    
    local function loadDeps(deps)
        if not deps then return end
        for _,proj in ipairs(deps) do
            addTask(TASK_loadProject, proj)
            addTask(TASK_loadDependencies, proj)
            addTask(TASK_validateProject, proj)
        end
    end
    
    -- Load any dependencies
    meta.deps = meta.deps or {}
    if meta.loaddev and meta.devdeps then
        table.move(meta.devdeps, 1, #meta.devdeps, #meta.deps+1, meta.deps)
    end
    loadDeps(meta.deps)
    
    -- Success
    return true
end

TASK_installProject = function(meta)
    local downloaded = false
    local ppath = (asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version).path
    
    -- Create project/version folder
    local fm = objc.NSFileManager.defaultManager
    if not fm:fileExistsAtPath_isDirectory_(ppath, true) then
        fm:createDirectoryAtPath_withIntermediateDirectories_attributes_error_(ppath, true, nil, nil)
    end
    
    local function install()
        print("Installing '" .. meta.name .. "-" .. meta.version .. "'")
        
        -- Extract .codea project from project.zip
        -- Open a new zip archive for reading
        local zip = Zip(asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/project.zip")
        
        -- Get all files in the zip
        local files = zip:listFiles()
        
        -- Extract all files
        for _,file in ipairs(files) do
            
            -- Ensure the dir exists
            local dir = ppath .. "/" .. file:match(".*/")
            if not fm:fileExistsAtPath_isDirectory_(dir, true) then
                fm:createDirectoryAtPath_withIntermediateDirectories_attributes_error_(dir, true, nil, nil)
            end
            
            -- Extract file
            local f = io.open(ppath .. "/" .. file, "wb")
            f:write(zip:readFile(file))
            f:close()
        end
        
        -- Close the zip file
        zip:close() 
        
        -- Add 'valid' flag file
        saveText(asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/valid.txt", "READY_TO_USE")
    
        return true
    end
    
    local url = "https://codeawebrepo.co.uk/" .. socket.url.escape(meta.name) .. "/" .. socket.url.escape(meta.version) .. "/project.zip"
    
    -- Get the headers only first
    http.request(url, function(res, code, headers)
        
        -- Download files in chunks to avoid memory issues
        local size = math.tointeger(headers['Content-Length'])
        local totalSize = size
        local pos = 0
        local file = io.open(ppath .. "/project.zip", "wb")
        local CHUNK_SIZE = 1 * 1024 * 1024 -- 1MB at a time
        
        local function nextChunk()
            local chunkSize = math.min(size, CHUNK_SIZE)
            size = size - chunkSize
            http.request(url, function(res)
                file:write(res)
                pos = pos + chunkSize
                --if progress_cb then progress_cb(pos/totalSize) end
                if size > 0 then
                    nextChunk()
                else
                    file:close()
                    downloaded = true
                    addTask(install)
                end
            end, function(err)
                file:close()
                error("Failed to download '" .. meta.name .. "-" .. meta.version .. "' (" .. err .. ")")
            end, { headers = { Range = string.format("bytes=%d-%d", pos, pos + chunkSize - 1)}})
        end
        nextChunk()
        
    end, function(err)
        error("Failed to download '" .. meta.name .. "-" .. meta.version .. "' (" .. err .. ")")
    end, { method = "HEAD" })
    
    -- Keep running the same task until the project is installed
    print("Downloading '" .. meta.name .. "-" .. meta.version .. "'")
    local function wait()
        if downloaded == false then
            addTask(wait)
            taskLoopYield()
        end
        -- Success
        return true
    end
    addTask(wait)
    
    -- Success
    return true
end

TASK_loadProject = function(meta)
    meta.version = meta.version or "__LOCAL__"
    
    -- TODO: Allow name overrides for use in 'require()'
    -- Use cached env if this project has already been loaded
    if ENVS[meta.name] and ENVS[meta.name][meta.version] ~= nil then
        local cmeta = ENVS[meta.name][meta.version]
        if cmeta.loadmain == meta.loadmain then
            meta.env = cmeta.env
            return true
        end
        -- TODO devdeps check too
    end
    
    print("Loading " .. meta.name)
    local ppath, plist
    if meta.version == "__LOCAL__" then
        -- Read plist
        plist = readText(asset.documents .. meta.name .. ".codea/Info.plist")
        if plist == nil then
            if (asset.documents .. meta.name .. ".codea") ~= nil then
                plist = "<key>Buffer Order</key><array><string>Main</string></array>"
            else
                error("Local project '" .. meta.name .. "' does not appear to be installed!")
            end
        end
        
        ppath = (asset.documents .. meta.name .. ".codea").path .. "/"
    else
        -- Remote project
        
        -- Read plist
        plist = readText(asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/" .. meta.name .. ".codea/Info.plist")
        if plist == nil then
            if (asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/" .. meta.name .. ".codea") ~= nil then
                plist = "<key>Buffer Order</key><array><string>Main</string></array>"
            else
                error("'" .. meta.name .. "' does not appear to be installed!", 0)
            end
        end
        ppath = (asset.documents.webrepolib_cache .. meta.name .. "/" .. meta.version .. "/" .. meta.name .. ".codea").path .. "/"
    end
    
    -- Create a new env for the project to load into
    local env, depenvs = createInternalEnv(meta)
    
    -- Auto import dependencies
    -- (The dependency is copied into the current project's env)
    for _,dep in pairs(meta.deps) do
        if dep.export then
            WRL.Env.import(dep.env, env)
        elseif dep.import or dep.import == nil then
            table.insert(depenvs, dep.env)
        end
    end
    
    -- Load project into environment
    local buffers = plist:match("<key>Buffer Order</key>%s*<array>(.-)</array>")
    for b in buffers:gmatch("<string>(.-)</string>") do
        if (b == "Main" and meta.loadmain ~= true) or (b == "WRL") then
            goto CONTINUE
        end
        
        -- Load from file
        local fn, err = loadfile(ppath .. b .. ".lua", nil, env)
        if fn == nil then
            error(err)
        end
                
        -- Execute file
        if (WRL.CrashableCall(fn) == false) then
            return false
        end
            
        ::CONTINUE::
    end
    
    -- Convert env variable to an external env (no metatable)
    env = createExternalEnv(env)
    
    -- Use environment
    meta.env = env
    
    -- Add env to ENVS table
    ENVS[meta.name] = ENVS[meta.name] or {}
    ENVS[meta.name][meta.version or "__LOCAL__"] = meta
    
    -- Success
    return true
end

TASK_go = function(meta)
    -- Nil internal refs
    WRL.Env = nil
    WRL.Tasks = nil
    
    -- Call OnLoad on every loaded project
    for _,versions in pairs(ENVS) do
        for _,meta in pairs(versions) do
            if meta.env.OnLoad then
                meta.env.OnLoad()
                -- Remove OnLoad() so it isn't repeated
                meta.env.OnLoad = nil
            end
        end
    end
    
    -- Import the root project to the global environment
    importEnv(meta.env)

    print("Done.") -- Celebrate?
    
    -- Run the setup function
    if type(setup) == "function" then setup() end
    
    -- Success
    return true
end

addTask(TASK_kickoff)

-- Override the require function to insert our
-- loaded dependencies.
local _require = require
function require(...)
    
    local args = {...}
    local name = args[1]
    local version = args[2]
    
    if name and ENVS[name] then
        if version and ENVS[name][version] then
            return ENVS[name][version][1]
        else
            for k,v in pairs(ENVS[name]) do
                return v[1]
            end
        end
    end
    
    return _require(...)
end

function import(...)
    local args = {...}
    local name = args[1]
    local version = args[2]
    
    local _cEnv -- env of caller
    local i = 2
    local info = debug.getinfo(i)
    while info ~= nil and _cEnv == nil do
        local func = info.func
        for i=1,info.nups do
            local k,v = debug.getupvalue(func, i)
            if k == "_ENV" then
                _cEnv = v
                break
            end
        end
        
        i = i + 1
        info = debug.getinfo(i)
    end
    
    -- Import the named project into the environment of the caller
    if name and ENVS[name] then
        if version and ENVS[name][version] then
            return WRL.Env.import(ENVS[name][version][1], _cEnv)
        else
            for k,v in pairs(ENVS[name]) do
                return WRL.Env.import(v[1], _cEnv)
            end
        end
    end
    
    error("Project not declared in WRL!")
end

end
------------------------------
-- Zip.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

----------------------------------------
-- Binary form of Zip library (speeds up loading & reduces size)
----------------------------------------
local ZIP_BIN_P1 = [==[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]==]
local ZIP_BIN_P2 = [==[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]==]
-- The b64 encoded binary above is split as Codea's editor seems to breakdown with very long single lines preventing the rendering of the text.
do -- Load the Zip library
    local mime = require("mime")
    ZIP_BIN = mime.unb64(ZIP_BIN_P1 .. ZIP_BIN_P2)
    load(ZIP_BIN)()
    ZIP_BIN = nil
    ZIP_BIN_P1 = nil
    ZIP_BIN_P2 = nil
end
end
------------------------------
-- LoadGuard.lua
------------------------------
do
-- Only load once
if WRL and WRL.loading then return end

-- Check all current project source files for load guards on the first lines
if asset.name ~= "WebRepoLib.codea" then  -- exluding WRL itself.
    
    -- Read plist
    local f = io.open((asset .. "Info.plist").path, "r")
    local plist = f:read("*a") f:close()
    ppath = asset.path .. "/"
    
    local guardCode = 
[[----------------- DO NOT REMOVE OR MOVE -----------------
-- Ensure Codea doesn't load this file automatically
-- This MUST be at the top of this file!
if WRL and not WRL.loading then return end
--------------- END DO NOT REMOVE OR MOVE ---------------
]]
    local guardCodePat = guardCode:gsub("%-", "%%-"):gsub("%(%)", "%%(%%)")
    
    local restart = false
    
    -- Check each source file
    local buffers = plist:match("<key>Buffer Order</key>%s*<array>(.-)</array>")
    for b in buffers:gmatch("<string>(.-)</string>") do

        -- Don't add to the _dep_Documents
        if b ~= "_dep_Documents:WebRepoLib" then
    
            -- Read source file
            f = io.open(ppath .. b .. ".lua", "r")
            local src = f:read("*a")
            f:close()
                
            -- Check for load guard
            if src:match("^" .. guardCodePat) == nil then
                -- Add load guard if not present and request restart
                f = io.open(ppath .. b .. ".lua", "w")
                f:write(guardCode .. src)
                f:close()
                restart = true
            end
        end
    end
    
    if restart then
        -- ****************************************************
        -- WebRepoLib automatically adds load guards at the
        -- beginning of project source files to ensure it
        -- controls the load order of files and dependencies.
        --
        -- When these guards are added, we need to close and
        -- re-open the project editor to ensure Codea can see
        -- the changes!
        error("Close & Re-open your project!")
        -- ****************************************************
    end
end
end
------------------------------
-- WRL.lua
------------------------------
do
-- Only load once
if WRL and not WRL.loading then return end

devdeps = {
    "Codea+docs"
}
end
