local websocket = require("websocket")
local base64 = require("base64")

local b64 = base64.b64
local unb64 = base64.unb64

local exports = {}

local Worker = class()
function Worker:init(websocket)
	self.ws = websocket
	self.receivers = {}
end

function Worker:send(msg)
	local payload = {
		data = b64(msg)
	}
	payload = json.encode(payload)
	self.ws:send(payload)
end

function Worker:recv()
	-- Block waiting for message
	local data, err = self.ws:recv()
	--assert(data ~= nil)
	local payload = json.decode(data)
	payload = unb64(payload)
	--assert(payload.data ~= nil)
	return payload.data
end

function Worker:newReceiver(callId)
	local worker = self
	
	local receiver = {
		hasResult = false
	}
	function receiver:waitForResult()
		while self.hasResult == false do
			worker:updateReceivers()
		end
		return self._result
	end
	function receiver:next(fn)
		self.listener = fn
	end
	
	self.receivers[callId] = receiver
	
	return receiver
end

function Worker:updateReceivers()
	while true do
		if self.ws:canread() then
			local data, err = self.ws:recv()
			if data then
				local payload = json.decode(data)
				local msg = unb64(payload).data
				
				local receiver = self.receivers[msg.id]
				receiver.hasResult = true
				receiver._result = msg.data
				
				-- Call listener
				if receiver.listener then
					receiver.listener(msg.data)
				end
			end
		else
			return
		end
	end
end

local WorkerManager = class()
function WorkerManager:init(port, webview)
	self.wv = webview
	
	-- Initialise the websocket server
	self.wss = websocket(port)
	
	-- Accept core client
	self.ws = self.wss:getClient()
	
	-- Our array of workers
	self.workers = {}
end

function WorkerManager:newWorker()
	-- Create a new js worker thread
	self.wv:exec([[
		let worker = new Worker("worker.js");
		
		// Worker -> Codea
		worker.onmessage = (e)=>{
			socket.send(e.data);
		};
	]])
	
	-- Get our new client connection directly to the worker
	local w = Worker(self.wss:getClient())
	table.insert(self.workers, w)
	
	-- Wait for the 'ready' signal
	assert(w:recv() == "_ready_")
	
	return w
end

function WorkerManager:update(limit)
	for i=1,limit do
		if self.ws:canread() then
			self.ws:recv(true)
		else
			break
		end
	end
	for _,worker in ipairs(self.workers) do
		worker:updateReceivers()
	end
	-- objc.warning("[Parallel] >" .. limit .. " received messages per frame.\nThis could affect performance.")
end

return WorkerManager