-- Contents:
--    Tween.lua
--    GlobalWrappers.lua
--    Thread.lua
--    Semaphore.lua
--    Async.lua
--    ThreadCompat.lua
--    Utils.lua
--    Debugger.lua
--    AutoDoc.lua

------------------------------
-- Tween.lua
------------------------------
do
-- tween.lua
-- v1.0.1 (2012-02)

-- Based on:
-- Enrique García Cota - enrique.garcia.cota [AT] gmail [DOT] com
-- tweening functions for lua
-- inspired by jquery's animate function

-- Modified for Codea by John Millard 2013-02
-- Modified for Codea+ by Oliver Steptoe 2022-04
-----------------------------------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------------------------------

tween = {}

-- private stuff

local tweens = {}

local function isCallable(f)
  local tf = type(f)
  if tf == 'function' then return true end
  if tf == 'table' then
    local mt = getmetatable(f)
    return (type(mt) == 'table' and type(mt.__call) == 'function')
  end
  return false
end

local function copyTables(destination, keysTable, valuesTable)
  valuesTable = valuesTable or keysTable
  for k,v in pairs(keysTable) do
    if type(v) == 'table' then
      destination[k] = copyTables({}, v, valuesTable[k])
    else
      destination[k] = valuesTable[k]
    end
  end
  return destination
end

local function checkSubjectAndTargetRecursively(subject, target, path)
  path = path or {}
  local targetType, newPath
  for k,targetValue in pairs(target) do
    targetType, newPath = type(targetValue), copyTables({}, path)
    table.insert(newPath, tostring(k))
    if targetType == 'number' then
      assert(type(subject[k]) == 'number', "Parameter '" .. table.concat(newPath,'/') .. "' is missing from subject or isn't a number")
    elseif targetType == 'userdata' then
      assert(type(subject[k]) == 'userdata', "Parameter '" .. table.concat(newPath,'/') .. "' is missing from subject or isn't a userdata")
    elseif targetType == 'table' then
      checkSubjectAndTargetRecursively(subject[k], targetValue, newPath)
    else
      assert(targetType == 'number', "Parameter '" .. table.concat(newPath,'/') .. "' must be a number or table of numbers")
    end
  end
end

local function getEasingFunction(easing)
  easing = easing or "linear"

  if type(easing) == 'table' then
    easing = easing['easing'] or "linear"
    end
    
  if type(easing) == 'string' then
    local name = easing
    easing = tween.easing[name]
    assert(type(easing) == 'function', "The easing function name '" .. name .. "' is invalid")
  end
  return easing
end

local function getloopFunction(loop)
  loop = loop or "once"
  
  if type(loop) == 'table' then
    loop = loop['loop'] or "once"
    
    if type(loop) == 'string' then
      local name = loop
      loop = tween.loop[name]
      assert(type(loop) == 'function', "The loop function name '" .. name .. "' is invalid")
    end
  else
    loop = tween.loop.once
  end

  return loop
end

local function checkStartParams(time, subject, target, options, callback)
  local easing = getEasingFunction(options)
  local loop = getloopFunction(options)

  assert(type(time) == 'number' and time > 0, "time must be a positive number. Was " .. tostring(time))
  local tsubject = type(subject)
  assert(tsubject == 'table' or tsubject == 'userdata', "subject must be a table or userdata. Was " .. tostring(subject))
  assert(type(target)== 'table', "target must be a table. Was " .. tostring(target))
  assert(isCallable(easing), "easing must be a function or functable. Was " .. tostring(easing))
  assert(isCallable(loop), "loop must be a function or functable. Was " .. tostring(loop))
  assert(callback==nil or isCallable(callback), "callback must be nil, a function or functable. Was " .. tostring(time))
  checkSubjectAndTargetRecursively(subject, target)

end

local function newTween(time, subject, target, options, callback, args)
  local self = {
    time = time,
    subject = subject,
    target = target,
    easing = getEasingFunction(options),
    loop = getloopFunction(options),
    callback = callback,
    args = args,
    initial = copyTables({}, target, subject),
    running = 0
  }
  tweens[self] = self
  return self
end

local function easeWithTween(self, subject, target, initial)
  local t,b,c,d

  for k,v in pairs(target) do
    if type(v)=='table' then
      easeWithTween(self, subject[k], v, initial[k])
    else
      t,b,c,d = self.running, initial[k], v - initial[k], self.time
      
      if self.loop then
        t = self.loop(t,d)
      end

      subject[k] = self.easing(t,b,c,d)
    end
  end
end

local function updateTween(self, dt)
    if Debugger.thread and self.ignoreDebug ~= true then return end
    self.running = self.running + dt
    easeWithTween(self, self.subject, self.target, self.initial)
end

local function hasExpiredTween(self)
  if self.loop == tween.loop.once then
    return self.running >= self.time
  else
    return false
  end
end

local function finishTween(self)
  self.overshoot = self.running - self.time
  self.running = self.overshoot

  tween.stop(self)

    if self.callback then
        if self.noAsyncCallback then
            self.callback(table.unpack(self.args))
        else
            Thread.__runAsync(self.callback, table.unpack(self.args))
        end
    end

  if self.next then
    self.next.initial = copyTables(self.next.initial, self.target, self.subject)
    tweens[self.next] = self.next
  end

end

local function resetTween(self)
  self.running = 0
  easeWithTween(self, self.subject, self.target, self.initial)
  --copyTables(self.subject, self.initial)
end

-- paths

local function pathFactory(subject, values)
    local len = #values
    
    assert(len >= 1, "Path length must be one or greater")
    
    local keys = {}
    for k,v in pairs(values[1]) do
        table.insert(keys,k)
    end
    
    local pathFunc = function(t,k)
        
        t = (1 + (t*(len-1)))
        
        local i2 = math.floor(t)
        local i1 = math.max(i2 - 1, 1)
        local i3 = math.min(i2 + 1, len)
        local i4 = math.min(i3 + 2, len)
        
        t = t - i2
        
        local p1 = values[i1][k]
        local p2 = values[i2][k]
        local p3 = values[i3][k]
        local p4 = values[i4][k]
        
        local t2 = t*t
        local t3 = t*t*t
        
        return 0.5 * ( (2*p2) + 
                       (-p1+p3)*t + 
                       (2*p1 - 5*p2 + 4*p3 - p4)*t2 + 
                       (-p1 + 3*p2 - 3*p3 + p4)*t3)
    end
    
    local proxy = {}
    local mt = 
    {
        __index = function(table,k)
            if k == 't' then
                return 0
            end
            return rawget(table,k)
        end,
        
        __newindex = function(table,k,v)
            if k == 't' then
                for i,key in pairs(keys) do
                    subject[key] = pathFunc(v,key)
                end
            else
                rawset(table,k,v)
            end
        end 
    }
    setmetatable(proxy,mt)
    return proxy
end



-- easing

-- Adapted from https://github.com/EmmanuelOga/easing. See LICENSE.txt for credits.
-- For all easing functions:
-- t = time == how much time has to pass for the tweening to complete
-- b = begin == starting property value
-- c = change == ending - beginning
-- d = duration == running time. How much time has passed *right now*

local pow = function(x, y) return x^y end
local sin, cos, pi, sqrt, abs, asin = math.sin, math.cos, math.pi, math.sqrt, math.abs, math.asin

-- linear
local function linear(t, b, c, d) return c * t / d + b end

-- quad
local function inQuad(t, b, c, d) return c * pow(t / d, 2) + b end
local function outQuad(t, b, c, d)
  t = t / d
  return -c * t * (t - 2) + b
end
local function inOutQuad(t, b, c, d)
  t = t / d * 2
  if t < 1 then return c / 2 * pow(t, 2) + b end
  return -c / 2 * ((t - 1) * (t - 3) - 1) + b
end
local function outInQuad(t, b, c, d)
  if t < d / 2 then return outQuad(t * 2, b, c / 2, d) end
  return inQuad((t * 2) - d, b + c / 2, c / 2, d)
end

-- cubic
local function inCubic (t, b, c, d) return c * pow(t / d, 3) + b end
local function outCubic(t, b, c, d) return c * (pow(t / d - 1, 3) + 1) + b end
local function inOutCubic(t, b, c, d)
  t = t / d * 2
  if t < 1 then return c / 2 * t * t * t + b end
  t = t - 2
  return c / 2 * (t * t * t + 2) + b
end
local function outInCubic(t, b, c, d)
  if t < d / 2 then return outCubic(t * 2, b, c / 2, d) end
  return inCubic((t * 2) - d, b + c / 2, c / 2, d)
end

-- quart
local function inQuart(t, b, c, d) return c * pow(t / d, 4) + b end
local function outQuart(t, b, c, d) return -c * (pow(t / d - 1, 4) - 1) + b end
local function inOutQuart(t, b, c, d)
  t = t / d * 2
  if t < 1 then return c / 2 * pow(t, 4) + b end
  return -c / 2 * (pow(t - 2, 4) - 2) + b
end
local function outInQuart(t, b, c, d)
  if t < d / 2 then return outQuart(t * 2, b, c / 2, d) end
  return inQuart((t * 2) - d, b + c / 2, c / 2, d)
end

-- quint
local function inQuint(t, b, c, d) return c * pow(t / d, 5) + b end
local function outQuint(t, b, c, d) return c * (pow(t / d - 1, 5) + 1) + b end
local function inOutQuint(t, b, c, d)
  t = t / d * 2
  if t < 1 then return c / 2 * pow(t, 5) + b end
  return c / 2 * (pow(t - 2, 5) + 2) + b
end
local function outInQuint(t, b, c, d)
  if t < d / 2 then return outQuint(t * 2, b, c / 2, d) end
  return inQuint((t * 2) - d, b + c / 2, c / 2, d)
end

-- sine
local function inSine(t, b, c, d) return -c * cos(t / d * (pi / 2)) + c + b end
local function outSine(t, b, c, d) return c * sin(t / d * (pi / 2)) + b end
local function inOutSine(t, b, c, d) return -c / 2 * (cos(pi * t / d) - 1) + b end
local function outInSine(t, b, c, d)
  if t < d / 2 then return outSine(t * 2, b, c / 2, d) end
  return inSine((t * 2) -d, b + c / 2, c / 2, d)
end

-- expo
local function inExpo(t, b, c, d)
  if t == 0 then return b end
  return c * pow(2, 10 * (t / d - 1)) + b - c * 0.001
end
local function outExpo(t, b, c, d)
  if t == d then return b + c end
  return c * 1.001 * (-pow(2, -10 * t / d) + 1) + b
end
local function inOutExpo(t, b, c, d)
  if t == 0 then return b end
  if t == d then return b + c end
  t = t / d * 2
  if t < 1 then return c / 2 * pow(2, 10 * (t - 1)) + b - c * 0.0005 end
  return c / 2 * 1.0005 * (-pow(2, -10 * (t - 1)) + 2) + b
end
local function outInExpo(t, b, c, d)
  if t < d / 2 then return outExpo(t * 2, b, c / 2, d) end
  return inExpo((t * 2) - d, b + c / 2, c / 2, d)
end

-- circ
local function inCirc(t, b, c, d) return(-c * (sqrt(1 - pow(t / d, 2)) - 1) + b) end
local function outCirc(t, b, c, d)  return(c * sqrt(1 - pow(t / d - 1, 2)) + b) end
local function inOutCirc(t, b, c, d)
  t = t / d * 2
  if t < 1 then return -c / 2 * (sqrt(1 - t * t) - 1) + b end
  t = t - 2
  return c / 2 * (sqrt(1 - t * t) + 1) + b
end
local function outInCirc(t, b, c, d)
  if t < d / 2 then return outCirc(t * 2, b, c / 2, d) end
  return inCirc((t * 2) - d, b + c / 2, c / 2, d)
end

-- elastic
local function calculatePAS(p,a,c,d)
  p, a = p or d * 0.3, a or 0
  if a < abs(c) then return p, c, p / 4 end -- p, a, s
  return p, a, p / (2 * pi) * asin(c/a) -- p,a,s
end
local function inElastic(t, b, c, d, a, p)
  local s
  if t == 0 then return b end
  t = t / d
  if t == 1  then return b + c end
  p,a,s = calculatePAS(p,a,c,d)
  t = t - 1
  return -(a * pow(2, 10 * t) * sin((t * d - s) * (2 * pi) / p)) + b
end
local function outElastic(t, b, c, d, a, p)
  local s
  if t == 0 then return b end
  t = t / d
  if t == 1 then return b + c end
  p,a,s = calculatePAS(p,a,c,d)
  return a * pow(2, -10 * t) * sin((t * d - s) * (2 * pi) / p) + c + b
end
local function inOutElastic(t, b, c, d, a, p)
  local s
  if t == 0 then return b end
  t = t / d * 2
  if t == 2 then return b + c end
  p,a,s = calculatePAS(p,a,c,d)
  t = t - 1
  if t < 0 then return -0.5 * (a * pow(2, 10 * t) * sin((t * d - s) * (2 * pi) / p)) + b end
  return a * pow(2, -10 * t) * sin((t * d - s) * (2 * pi) / p ) * 0.5 + c + b
end
local function outInElastic(t, b, c, d, a, p)
  if t < d / 2 then return outElastic(t * 2, b, c / 2, d, a, p) end
  return inElastic((t * 2) - d, b + c / 2, c / 2, d, a, p)
end

-- back
local function inBack(t, b, c, d, s)
  s = s or 1.70158
  t = t / d
  return c * t * t * ((s + 1) * t - s) + b
end
local function outBack(t, b, c, d, s)
  s = s or 1.70158
  t = t / d - 1
  return c * (t * t * ((s + 1) * t + s) + 1) + b
end
local function inOutBack(t, b, c, d, s)
  s = (s or 1.70158) * 1.525
  t = t / d * 2
  if t < 1 then return c / 2 * (t * t * ((s + 1) * t - s)) + b end
  t = t - 2
  return c / 2 * (t * t * ((s + 1) * t + s) + 2) + b
end
local function outInBack(t, b, c, d, s)
  if t < d / 2 then return outBack(t * 2, b, c / 2, d, s) end
  return inBack((t * 2) - d, b + c / 2, c / 2, d, s)
end

-- bounce
local function outBounce(t, b, c, d)
  t = t / d
  if t < 1 / 2.75 then return c * (7.5625 * t * t) + b end
  if t < 2 / 2.75 then
    t = t - (1.5 / 2.75)
    return c * (7.5625 * t * t + 0.75) + b
  elseif t < 2.5 / 2.75 then
    t = t - (2.25 / 2.75)
    return c * (7.5625 * t * t + 0.9375) + b
  end
  t = t - (2.625 / 2.75)
  return c * (7.5625 * t * t + 0.984375) + b
end
local function inBounce(t, b, c, d) return c - outBounce(d - t, 0, c, d) + b end
local function inOutBounce(t, b, c, d)
  if t < d / 2 then return inBounce(t * 2, 0, c, d) * 0.5 + b end
  return outBounce(t * 2 - d, 0, c, d) * 0.5 + c * .5 + b
end
local function outInBounce(t, b, c, d)
  if t < d / 2 then return outBounce(t * 2, b, c / 2, d) end
  return inBounce((t * 2) - d, b + c / 2, c / 2, d)
end

tween.easing = {
  linear = linear,
  quadIn    = inQuad,    quadOut    = outQuad,    quadInOut    = inOutQuad,    quadOutIn    = outInQuad,
  cubicIn   = inCubic,   cubicOut   = outCubic,   cubicInOut   = inOutCubic,   cubicOutIn   = outInCubic,
  quartIn   = inQuart,   quartOut   = outQuart,   quartInOut   = inOutQuart,   quartOutIn   = outInQuart,
  quintIn   = inQuint,   quintOut   = outQuint,   quintInOut   = inOutQuint,   quintOutIn   = outInQuint,
  sineIn    = inSine,    sineOut    = outSine,    sineInOut    = inOutSine,    sineOutIn    = outInSine,
  expoIn    = inExpo,    expoOut    = outExpo,    expoInOut    = inOutExpo,    expoOutIn    = outInExpo,
  circIn    = inCirc,    circOut    = outCirc,    circInOut    = inOutCirc,    circOutIn    = outInCirc,
  elasticIn = inElastic, elasticOut = outElastic, elasticInOut = inOutElastic, elasticOutIn = outInElastic,
  backIn    = inBack,    backOut    = outBack,    backInOut    = inOutBack,    backOutIn    = outInBack,
  bounceIn  = inBounce,  bounceOut  = outBounce,  bounceInOut  = inOutBounce,  bounceOutIn  = outInBounce,
}

-- ping pong

local function round(num)
  if num >= 0 then return math.floor(num+.5)
  else return math.ceil(num-.5) end
end

local function triangleWave(x)
  return 1 - 2 * math.abs(round(0.5 * x) - 0.5*x)
end

local function frac(x)
  local a,b = math.modf(x)
  return b
end

local function sawtooth(x)
  return frac(0.5 * x)
end

local function forever(t,d)
  return sawtooth(t/(d/2)) * d
end

local function pingpong(t,d)
    return (1-triangleWave(t/d)) * d
end

local function once(t,d)
    return math.min(t,d)
end

tween.loop = 
{
    once = once,
    forever = forever,
    pingpong = pingpong
}

-- public functions

function tween.start(time, subject, target, options, callback, ...)
  checkStartParams(time, subject, target, options, callback)
  return newTween(time, subject, target, options, callback, {...})
end

function tween.sequence(...)
    local n = select('#', ...)
    local arg = {...}
    assert(n > 0, "a sequence must consist of at least one tween")

    local head = arg[1]
    local tail = arg[n]

    for i,id in pairs(arg) do
        local tw = tweens[id]
        
        tw.head = head
        tw.tail = tail

        if i < n then
            tw.next = arg[i+1]
        end

        -- remove all except the first tween
        if i > 1 and tw then
            tweens[id] = nil
        end
    end

    return arg[1]
end

function tween.path(time, subject, target, options, callback, ...)
  -- need to check that target is an array and contains valid keys
  return newTween(time, pathFactory(subject, target) , {t=1}, options, callback, {...})
end

function tween.delay(duration, callback, ...)
    local t = {}
    return tween(duration, t, {}, tween.easing.linear, callback, ...)
end

setmetatable(tween, { __call = function(t, ...) return tween.start(...) end })

function tween.play(id)
    tweens[id]=id
end

function tween.reset(id)
  local tw = tweens[id]
  if tw then
    resetTween(tw)
    tween.stop(tw)
  end
end

function tween.resetAll(id)
  for _,tw in pairs(tweens) do 
    resetTween(tw)
    --copyTables(tw.subject, tw.initial)
  end
  tween.stopAll()
end

function tween.stop(id)
  -- this tween object is the head of a sequence, stop all connected tweens
  if id.head and id.head == id then
    local next = id.next
    while next ~= nil do
      tweens[next]=nil
      next = next.next
    end
  end

  if id~=nil then 
    tweens[id]=nil
  end
end

function tween.stopAll()
  tweens = {}
end

function tween.hasExpired(id)
  local tw = tweens[id]
  if tw then
    return hasExpiredTween(id)
  end
  return true
end

function tween.update(dt)
    if dt <= 0 then
        return
    end
    
    local expired = {}
    for _,t in pairs(tweens) do
        updateTween(t, dt)
        if hasExpiredTween(t) then
            table.insert(expired, t)
        end
    end
    for i=1, #expired do finishTween(expired[i]) end
end

function tween.count()
  local c = 0
  for _,v in pairs(tweens) do
    c = c + 1
  end
  return c
end

-- Executes a function every 'period' seconds
-- 
-- Period is not accurate & is bound to frame
-- rate. The callback will be called a maximum
-- of once per frame.
--
-- If for instance your period is equivalent
-- to 1.5x framerate, when the period elapses
-- mid-frame the callback will not be executed
-- until the following frame.
function tween.interval(period, callback)
    local loop = tween.delay(period)
    loop.callback = function()
        
        -- Call the interval callback
        callback(loop)
        
        -- Did the callback call stop?
        if loop.forceStop then
            return
        end
        
        -- Execute as accurately as we can
        -- so account for any overshoot of timing
        -- in the current invocation
        loop.time = math.max(0, period - loop.overshoot)
            
        -- Restart our tween
        tween.reset(loop)
        tween.play(loop)
    end
    
    function loop:stop()
        tween.stop(self)
        self.forceStop = true
    end
    
    return loop
end

end
------------------------------
-- GlobalWrappers.lua
------------------------------
do
local gmt = getmetatable(_G)
if gmt and gmt.___CODEA_PLUS then
    return
end
assert(gmt == nil, "Global metatable modifications are not compatible with 'Codea+'")
    
    
local wrappers = {}
    
gmt = {
    __index = function(t, k)
        local w = wrappers[k]
        if w then
            return function(...)
                w.wfn(w.fn, ...)
            end
        end
    end,
    __newindex = function(t, k, v)
        if wrappers[k] then
            wrappers[k].fn = v
        else
            rawset(t, k, v)
        end
    end,
    ___CODEA_PLUS = true
}
setmetatable(_G, gmt)

function __wrapGlobal(name, fn)
    wrappers[name] = {
        wfn = fn,
        fn = nil
    }
end

function __getWrappedGlobal(name)
    return wrappers[name].fn
end

end
------------------------------
-- Thread.lua
------------------------------
do
local socket = require('socket')

-- Thread management values
local threads = {}
local current_thread = nil
local sbFlushImage = image(1,1)
local async_queue = {}

-- Thread management task
local managementTask = tween.interval(0.001, function()
    
    -- Reset per frame flags
    for _,thread in ipairs(threads) do
        thread.done_this_frame = false
    end
    
    -- Keep running while threads are still executing
    local thread_ran = true
    while thread_ran do
        
        -- Reset flag
        thread_ran = false
        
        local deadThreads = {}
    
        for i,thread in ipairs(threads) do
            
            local dbg_thread = Debugger.thread
            
            -- Is the coroutine dead?
            if coroutine.status(thread.co) == "dead" then
                table.insert(deadThreads, i)
                
            -- Has the thread finished for this frame?
            elseif thread.done_this_frame then
                
            -- Ignore this thread if it's sleeping
            elseif thread.sleep_end and socket.gettime() < thread.sleep_end then
                
                -- If the sleeping thread is 'blocking' pretend we ran
                if thread.blocking then
                    thread_ran = true
                end
                
            -- If we're debugging, only run the debugged thread
            elseif dbg_thread ~= nil and thread ~= dbg_thread then
                
            -- All checks have passed. Resume the thread!
            else
                -- Track the current thread
                current_thread = thread
                thread_ran = true
                
                local runMainRet = {}
                while true do
                    local res = table.pack(coroutine.resume(thread.co, table.unpack(runMainRet)))
                    if not res[1] then
                        -- Propagate the error
                        error(string.format("\n%s\n\n%s\n\nIGNORE TRACE BELOW:", res[2], debug.traceback(thread.co)))
                    elseif res[2] then
                        -- Call the value returned from the resume
                        -- and resume again.
                        -- This is how a thread forces something to
                        -- run on the main thread.
                        current_thread = nil
                        runMainRet = table.pack(res[2](select(3, table.unpack(res))))
                        current_thread = thread
                    else
                        -- Pause execution
                        break
                    end
                end
                
                current_thread = nil
            end
        end
        
        -- Remove dead threads
        for i=#deadThreads,1,-1 do
            table.remove(threads, deadThreads[i])
        end
    end
        
    -- Render an offscreen sprite to flush the
    -- spritebatcher.
    -- This ensures sprite rendering done on
    -- threads is actually visible on screen.
    -- A wild guess is that the batcher is only
    -- flushed after the draw() call, and not
    -- after tween handling.
    spriteMode(CORNER)
    sprite(sbFlushImage, -1, -1)
end)
managementTask.noAsyncCallback = true
managementTask.ignoreDebug = true

-- Thread functions
function yield(fast)
    if not coroutine.isyieldable() then
        error("Unable to yield!")
    end
    current_thread.done_this_frame = (fast == nil or fast == false)
    coroutine.yield()
end

function yieldframe()
    yield()
end


-- Thread class
Thread = class()
Thread.__name = "Codea+ Thread"
function Thread:init(fn, name)
    self.co = coroutine.create(fn)
    self.blocking = false
    self.name = name
    
    table.insert(threads, self)
end

function Thread.runOnMain(func, ...)
    if coroutine.isyieldable() then
        return coroutine.yield(func, ...)
    end
    return func(...)
end

function Thread.__runAsync(func, ...)
    if func == nil or Debugger.thread then return end
    local params = {...}
    for i,p in ipairs(params) do
        -- Convert userdata param to a table
        if type(p) == "userdata" and p.___getters then
            local np = {}
            for k,fn in pairs(p.___getters) do
                np[k] = fn(p)
            end
            params[i] = np
        end
    end
    table.insert(async_queue, { 
        fn = func,
        params = params
    })
end

-- Threaded callbacks
function Thread.callback(fn, ignoreDebugger)
    return function(...)
        if fn == nil or (Debugger.thread and not ignoreDebugger) then return end
        table.insert(async_queue, { 
            fn = fn,
            params = { ... }
        })
    end
end

function Thread.wrapMain(fn)
    return function(...)
        return Thread.runOnMain(fn, ...)
    end
end

function Thread.current()
    return current_thread
end

function Thread.allThreads()
    return threads
end

function Thread.sleep(duration)
    current_thread.sleep_end = socket.gettime() + duration
    yield()
end

__wrapGlobal("touched", Thread.__runAsync)
__wrapGlobal("hover", Thread.__runAsync)
__wrapGlobal("scroll", Thread.__runAsync)
__wrapGlobal("pinch", Thread.__runAsync)
__wrapGlobal("keyboard", Thread.__runAsync)
-- __wrapGlobal("willClose", Thread.__runAsync) -- Don't wrap this

-- Async callback thread
Thread(function()
    
    -- We're a blocking thread as callbacks should
    -- be handled in the same frame as they were issued
    current_thread.blocking = true
    
    while true do
        for _,cb in ipairs(async_queue) do
            cb.fn(table.unpack(cb.params))
        end
        async_queue = {}
        
        yieldframe()
    end
    
end)

-- These should do nothing
__wrapGlobal("setup", function() end)
__wrapGlobal("draw", function() end)

-- Thread to call setup() & draw()
Thread(function()
    
    -- We're a blocking thread
    current_thread.blocking = true
    
    -- Call setup() once
    local fn = __getWrappedGlobal("setup")
    if fn then fn() end
    
    -- Call draw() repeatedly
    while true do
        local fn = __getWrappedGlobal("draw")
        if fn then fn() end
        yieldframe()
    end
end)
end
------------------------------
-- Semaphore.lua
------------------------------
do
Semaphore = class()

function Semaphore:init()
    self.count = 0
    self.waiting = 0
end

function Semaphore:signal()
    self.count = self.count + (count or 1)
end

function Semaphore:signalAll()
    self.count = self.count + self.waiting
end

function Semaphore:wait(fast)
    self.waiting = self.waiting + 1
    while self.count == 0 do
        yield(fast)
    end
    self.count = self.count - 1
    self.waiting = self.waiting - 1
end

end
------------------------------
-- Async.lua
------------------------------
do
Promise = class()
local Chain = class()

Promise.__name = "Promise"
function Promise:init(func)
    
    -- No results
    self.val = nil
    self.err = nil
    
    self.sem = Semaphore()
    
    -- Chained functions
    self.chains = {}
    
    -- Run the provided function on a separate thread
    if func then
        Thread(function()
            func(function(...)
                self:resolve(...)
            end,
            function(err)
                self:reject(err)
            end)
        end)
    end
end

function Promise:resolve(...)
    self.val = {...}
        
    -- Pass along promise chain
    for _,chain in ipairs(self.chains) do
        chain:fulfill(self.val)
    end
        
    -- Resume threads that are waiting on us
    self.sem:signalAll()
end

function Promise:reject(err)
    self.err = err
    
    -- Pass along promise chain
    for _,chain in ipairs(self.chains) do
        chain:reject(err)
    end
    
    -- Resume threads that are waiting on us
    self.sem:signalAll()
end

function Promise:thenDo(onFulfilled, onRejected)
    local chain = Chain(onFulfilled, onRejected)
    table.insert(self.chains, chain)
    
    if self.val then
        chain:fulfill(self.val)
    elseif self.err then
        chain:reject(self.err)
    end
    
    return chain.promise
end

function Promise:catch(onRejected)
    return self:thenDo(nil, onRejected)
end

function Promise:finally(onFinally)
    local fn = function()
        return onFinally()
    end
    return self:thenDo(fn, fn)
end

function Promise.all(promises)
    local pr = Promise()
    
    local count = #promises
    local res = {}
    
    local function onRejected(err)
        pr:reject(err)
    end
    
    for i,p in ipairs(promises) do
        p:thenDo(function(...)
            -- If not rejected already
            if pr.err == nil then
                res[i] = {...} -- add the result
                
                -- Resolve the 'all' promise if
                -- all promises have resolved.
                count = count - 1
                if count == 0 then
                    pr:resolve(res)
                end
            end
        end, onRejected)
    end
    
    return pr
end

function Promise:await(fast)
    if self.val == nil and self.err == nil then
        self.sem:wait(fast)
    end
    if self.err then error(self.err:gsub("^.-:[0-9]+: ", ""), 2) end
    if self.val then return table.unpack(self.val) end
end

local function identity(arg)
    return arg
end

local function thrower(err)
    error(err:gsub("Async:[0-9]+: ", "")) -- Strip error line
end

function Chain:init(onFulfilled, onRejected)
    self.promise = Promise()
    self.onFullfilled = onFulfilled or identity
    self.onRejected = onRejected or thrower
end

function Chain:fulfill(val)
    Thread(function()
        local r = table.pack(pcall(self.onFullfilled, table.unpack(val)))
        if r[1] then
            self.promise:resolve(select(2, table.unpack(r)))
        else
            self.promise:reject(r[2])
        end
    end)
end

function Chain:reject(err)
    Thread(function()
        local r = table.pack(pcall(self.onRejected, err))
        if r[1] then
            self.promise:resolve(select(2, table.unpack(r)))
        else
            self.promise:reject(r[2])
        end
    end)
end

function async(func)
    return function(...)
        local params = {...}
        return Promise(function(resolve, reject)
            local res = table.pack(pcall(func, table.unpack(params)))
            if res[1] == true then
                -- Resolve & return all results
                resolve(select(2, table.unpack(res)))
            else
                -- Reject and return error message
                reject(res[2])
            end
        end)
    end
end

function await(promise, fast)
    return promise:await(fast)
end

end
------------------------------
-- ThreadCompat.lua
------------------------------
do
-- Some of Codea's builtin functions or objects
-- work incorrectly from within a coroutine.
-- The functions below hijack the metatable's of such
-- objects to ensure that they are only accessed from
-- the main thread (not a coroutine)

local wrapFunction = nil
local wrapTable = nil
local wrapUserdata = nil

local function wrap(lazy, ...)
    local res = {}
    
    for _,obj in ipairs({...}) do
        local t = type(obj)
        if obj == _G then -- Don't wrap the global table
            table.insert(res, obj)
        elseif t == "function" then
            table.insert(res, wrapFunction(obj))
        elseif t == "table" then
            table.insert(res, wrapTable(obj, lazy))
        elseif t == "userdata" then
            table.insert(res, wrapUserdata(obj))
        else
            table.insert(res, obj)
        end
    end
    
    return table.unpack(res)
end

wrapFunction = function(fn)
    return Thread.wrapMain(fn)
end

wrapTable = function(t, lazy)
    
    if lazy then
        
        -- Original metatable
        local omt = getmetatable(t)
        
        -- If we've already been converted, return table
        if omt and omt.___mainAccess then return t end
        
        -- Create a shallow clone of the table
        -- and clear the original to force it into
        -- our new metatable below
        local clone = {}
        for k, v in pairs(t) do
            clone[k] = v
            rawset(t, k, nil)
        end
        for k, v in ipairs(t) do
            clone[k] = v
            rawset(t, k, nil)
        end
        
        -- If the original metatable IS the table,
        -- the clone should do the same.
        if omt == t then
            setmetatable(clone, clone)
            omt = clone -- Consider the clone the new original mt
        else
            setmetatable(clone, omt)
        end
        
        -- Our new metatable
        local mt = {
            ___lazyObj = clone,
            ___mainAccess = true,
            
            __index = Thread.wrapMain(function(t, k)
                return wrap(true, clone[k])
            end),
            
            __newindex = Thread.wrapMain(function(t, k, v)
                clone[k] = v
            end),
            
            
        }
        
        setmetatable(t, mt)
    else
        
        -- Get our metatable
        local mt = getmetatable(t)
        
        -- Get our lazy object
        local lazyObj = mt.___lazyObj
        
        -- Copy all the lazy object values back
        -- into our table (wrapped this time)
        for k, v in pairs(lazyObj) do
            rawset(t, k, wrap(true, v))
        end
        for k, v in ipairs(lazyObj) do
            rawset(t, k, wrap(true, v))
        end
        
        -- We don't need our lazy object any more
        mt.___lazyObj = nil
    end
    
    return t
end

wrapUserdata = function(ud)
    local mt = getmetatable(ud)
            
    -- No metatable? Just add the userdata obj
    -- or already converted?
    if mt == nil or rawget(mt, "___mainAccess") == true then return ud end
    
    -- Set the 'converted' flag            
    rawset(mt, "___mainAccess", true)
    
    -- Custom __index function
    local __index = mt.__index
    mt.__index = function(t, k)
        -- '__cache' gets special non-wrapped treatment
        -- due to some Codea internals likely accessing
        -- its values directly using rawget.
        if k == "__cache" then
            return __index(t, k)
        end
        
        return wrap(true, Thread.runOnMain(__index, t, k))
    end
    
    -- Wraps the named metamethod in order to force it
    -- to execute on the main thread.
    local function wrapmm(name)
        local mm = rawget(mt, name)
        if type(mm) == "function" then
            -- Force it to execute on the main thread
            -- & ensure return values are also wrapped
            rawset(mt, name, function(...)
                return wrap(true, Thread.runOnMain(mm, ...))
            end)
        elseif mm then
            -- Ensure the value is adequately thread safe
            rawset(mt, name, wrap(true, mm))
        end
    end
    
    wrapmm("__add")
    wrapmm("__sub")
    wrapmm("__mul")
    wrapmm("__div")
    wrapmm("__mod")
    wrapmm("__pow")
    wrapmm("__unm")
    wrapmm("__idiv")
    wrapmm("__band")
    wrapmm("__bor")
    wrapmm("__bxor")
    wrapmm("__bnot")
    wrapmm("__shl")
    wrapmm("__shr")
    wrapmm("__add")
    wrapmm("__concat")
    wrapmm("__len")
    wrapmm("__eq")
    wrapmm("__lt")
    wrapmm("__le")
    wrapmm("__newindex")
    wrapmm("__call")
    wrapmm("__name")
    wrapmm("__close")
                
    return ud
end

-- Helper function
function accessOnMain(obj)
    return wrap(true, obj)
end

-- Override pairs & ipairs to ensure objects
-- with access forced to the main thread are
-- fully populated before iterating
local _pairs = pairs
function pairs(t)
    local mt = getmetatable(t)
    if mt and rawget(mt, "___lazyObj") then
        wrap(false, t)
    end
    return _pairs(t)
end

local _ipairs = ipairs
function ipairs(t)
    local mt = getmetatable(t)
    if mt and rawget(mt, "___lazyObj") then
        wrap(false, t)
    end
    return _ipairs(t)
end

-- Thread compatibility
function useThreadCompatLayer()
    layout = accessOnMain(layout)
    viewer = accessOnMain(viewer)
    asset = accessOnMain(asset)
end

-- Use the compat layer by default
useThreadCompatLayer()

end
------------------------------
-- Utils.lua
------------------------------
do
-- Various functions overridden to run on the main thread.
-- If this is not done, most of these functions will simply
-- not work at all from a Thread.

local http_request = http.request
http.request = function(url, success, fail, parameters)
    Thread.runOnMain(http_request, url, Thread.callback(success, true), Thread.callback(fail, true), parameters)
end

function http.requestProm(url, parameters)
    return Promise(function(resolve, reject)
        http.request(url, resolve, reject, parameters)
    end)
end

function http.requestSync(url, parameters)
    
    local result = nil
    local sem = Semaphore()
    
    Thread.runOnMain(http_request, url,
        function(response, status, headers) -- Success
            result = { response, status, headers }
            sem:signal()
        end,
        function(err) -- Fail
            result = { nil, err }
            sem:signal()
        end, parameters)
    
    sem:wait()
    
    return table.unpack(result)
end

local terminators = {}
function registerTerminator(func)
    table.insert(terminators, func)
end
function willClose()
    for _,term in ipairs(terminators) do
        term()
    end
end

end
------------------------------
-- Debugger.lua
------------------------------
do
Debugger = {
    thread = nil,
    debug_threads = false,
}

local saved_values = {}
local saved_viewer_mode = nil

local info = {
    traceback = {},
    locals = {},
    upvalues = {}
}

local function gatherDebugInfo()
    
    local traceback = info.traceback
    local locals = info.locals
    local upvalues = info.upvalues
    
    local level = 1
    while true do
        local info = debug.getinfo(Debugger.thread.co, level + 2, "Sl")
        if not info then break end
        
        locals[level] = {}
        upvalues[level] = {}
        
        if info.what == "C" then   -- is a C function?
            table.insert(traceback, string.format("[%d] C function", level))
            
        else   -- a Lua function
            table.insert(traceback, string.format("[%d]%s:%d", level, info.short_src, info.currentline))
            
            -- Gather locals
            local i = 1
            while true do
                local name, val = debug.getlocal(Debugger.thread.co, level + 2, i)
                if not name then break end
                locals[level][name] = { val, i }
                i = i + 1
            end
            
            -- Gather upvalues
            local func = debug.getinfo(Debugger.thread.co, level + 2).func
            i = 1
            while true do
                local name, val = debug.getupvalue(func, i)
                if not name then break end
                upvalues[level][name] = { val, i }
                i = i + 1
            end
            
        end
        level = level + 1
    end
    
    info.traceback = table.concat(traceback, "\n")
end

-- Continue
local function continue()
    Debugger.thread = nil
    _G.c = saved_values.c
    _G.l = saved_values.l
    _G.sl = saved_values.sl
    _G.t = saved_values.t
    
    
    -- Clear traceback, locals & upvalues
    info.traceback = {}
    info.locals = {}
    info.upvalues = {}
end

-- Locals
local function locals(name, level)
    
    -- Default to display first stack frame
    name = name or 1
    
    if type(name) == "string" then
        if level == nil then
            
            for i=1,#info.locals do
                
                -- Check locals first
                local v = info.locals[i][name]
                if v == nil then
                    -- Then upvalues
                    v = info.upvalues[i][name]
                end
                
                if v then
                    v = v[1]
                    local typ = type(v)
                    if typ == "table" or typ == "function" then
                        print(name .. " = " .. tostring(v))
                    else
                        print(name .. " = " .. typ .. ": " .. tostring(v))
                    end
                    return v
                end
            end
            
            objc.warning("No local '" .. name .. "' in any stack frame")
        else
            if level > #info.locals then
                objc.warning("Invalid stack frame: " .. level)
                return
            end
            
            -- Check locals first
            local v = info.locals[level][name]
            if v == nil then
                -- Then upvalues
                v = info.upvalues[level][name]
            end
            
            if v then
                v = v[1]
                local typ = type(v)
                if typ == "table" or typ == "function" then
                    print(name .. " = " .. tostring(v))
                else
                    print(name .. " = " .. typ .. ": " .. tostring(v))
                end
                return v
            else
                objc.warning("No local '" .. name .. "' in stack frame " .. level)
            end
        end
            
    elseif type(name) == "number" then
        if name > #info.locals then
            objc.warning("Invalid stack frame: " .. name)
            return
        end
        
        local out = {"Locals (level " .. name .. "):"}
        for k,v in pairs(info.locals[name]) do
            
            v = v[1]
            local typ = type(v)
            
            if typ == "table" or typ == "function" then
                table.insert(out, k .. " = " .. tostring(v))
            else
                local vs = tostring(v)
                if string.len(vs) > 32 then
                    vs = "..."
                end
                
                table.insert(out, k .. " = " .. typ .. ": " .. vs)
            end
        end
        
        -- Append upvalues
        for k,v in pairs(info.upvalues[name]) do
            v = v[1]
            local typ = type(v)
            
            if typ == "table" or typ == "function" then
                table.insert(out, k .. " = " .. tostring(v))
            else
                local vs = tostring(v)
                if string.len(vs) > 32 then
                    vs = "..."
                end
                
                table.insert(out, k .. " = " .. typ .. ": " .. vs)
            end
        end
        
        print(table.concat(out, "\n"))
    else
        objc.warning("l: Invalid parameter type. Expects string or number.")
    end
end

local function setlocal(name, value, level)
    
    if level == nil then
        
        for i=1,#info.locals do
            -- Check locals first
            local v = info.locals[i][name]
            
            if v then
                -- Set the new value
                debug.setlocal(Debugger.thread.co, i + 2, v[2], value)
                v[1] = value
                print("Local '" .. name .. "' set in stack frame " .. i)
                return
            else
                -- Then upvalues
                v = info.upvalues[i][name]
                if v then
                    -- Set the new value
                    local func = debug.getinfo(Debugger.thread.co, i + 2).func
                    debug.setupvalue(func, v[2], value)
                    v[1] = value
                    print("Local '" .. name .. "' set in stack frame " .. i)
                    return
                end
            end
        end
            
        objc.warning("No local '" .. name .. "' in any stack frame")
    else
        if level > #info.locals then
            objc.warning("Invalid stack frame: " .. level)
            return
        end
        
        -- Check locals first
        local v = info.locals[level][name]
            
        if v then
            -- Set the new value
            debug.setlocal(Debugger.thread.co, level, v[2], value)
            v[1] = value
            print("Local '" .. name .. "' set in stack frame " .. level)
            return
        else
            -- Then upvalues
            v = info.upvalues[level][name]
            if v then
                -- Set the new value
                local func = debug.getinfo(Debugger.thread.co, level + 2).func
                debug.setupvalue(func, v[2], value)
                v[1] = value
                print("Local '" .. name .. "' set in stack frame " .. level)
                return
            end
        end
        
        objc.warning("No local '" .. name .. "' in stack frame " .. level)
    end
end

local function traceback()
    print("Trace:\n" .. info.traceback)
end

local function launch_debugger()
    Debugger.thread = Thread.current()
    saved_values.c = _G.c
    saved_values.l = _G.l
    saved_values.sl = _G.sl
    saved_values.t = _G.t
    _G.c = continue
    _G.l = locals
    _G.sl = setlocal
    _G.t = traceback
    
    -- Show sidepanel
    saved_viewer_mode = viewer.mode
    viewer.mode = OVERLAY
    
    -- Pause physics
    physics.pause()
end

function dbg(condition)
    if condition == nil or condition == true then
        launch_debugger()
        gatherDebugInfo()
        
        -- Capture the current screen image
        local screen_image = viewer.snapshot()
        
        -- Print backtrace
        print("Debug break:\n" .. info.traceback)
        
        -- Yield until the debugger deactivates itself
        while Debugger.thread ~= nil do
            yieldframe()
            
            -- Draw the current screen so we don't flicker
            pushStyle()
            spriteMode(CORNER)
            sprite(screen_image, 0, 0, WIDTH, HEIGHT)
            popStyle()
        end
        
        -- Restore sidepanel mode
        viewer.mode = saved_viewer_mode
    
        -- Resume physics
        physics.resume()
    end
end

end
------------------------------
-- AutoDoc.lua
------------------------------
do

local function GenerateHTMLDoc()
    local tabs = listProjectTabs()
    
    for _,tab in ipairs(tabs) do
        
        -- Read the source
        local f = io.open(asset.path .. "/" .. tab .. ".lua", "r")
        local src = f:read("*a") f:close()
        
        for func in src:gmatch("%-%-%s*@function%s*(.-)%(") do
            print(func)
        end
    end
    
    return [[
    <!DOCTYPE html>
<html>
<body style="background-color:white;">

<h1>This is a heading</h1>
<p>This is a paragraph.</p>

</body>
</html>
    ]]
end

function AutoDoc()
    
    viewer.mode = FULLSCREEN
    
    -- Open the WebView
    local wv = WebView()
    wv:loadHTML(GenerateHTMLDoc())
    wv:show()
    
    -- Do not return
    while true do
        yield()
    end
end

end
