-- The tools used to apply voxels in the editor

TOOL_ADD = 1
TOOL_REPLACE = 2 
TOOL_ERASE = 3
TOOL_GET = 4

TOOL_TYPE_POINT = 1
TOOL_TYPE_LINE = 2
TOOL_TYPE_BOX = 3
TOOL_TYPE_FLOOD = 4

TOOL_STATE_IDLE = 1
TOOL_STATE_DRAG = 2

TOOL_UNIT_TYPE = "Solid"

Tool = class()

function Tool:init()
    self.state = TOOL_STATE_IDLE
    self.toolMode = TOOL_ADD
    self.toolType = TOOL_TYPE_BOX
    self.toolUnitType = TOOL_UNIT_TYPE
    self.toolColor = color(243, 243, 244)
    self.mirror = {x = false, y = false, z = false}
end

function Tool:touched(touch)    
    if #snapshots > 0 then
        volume:loadSnapshot(snapshots[#snapshots])
    end
    
    -- raycast() 在 Helper 标签页中定义
    local coord, id, face = raycast(touch.x, touch.y, false)
    
    if coord then
        if self.toolMode == TOOL_ADD then
            coord = coord + face
        end   
    end  
    
    if self.toolMode == nil then
        return false
    end
    
    if coord and touch.state == BEGAN and self.state == TOOL_STATE_IDLE then
        if #viewer.touches > 0 then return false end
        self.startCoord = coord
        self.endCoord = coord
        self.state = TOOL_STATE_DRAG
        self.points = {}
        table.insert(self.points, coord)
        self:apply()
        return true
    elseif touch.state == MOVING and self.state == TOOL_STATE_DRAG then
        if coord then
            self.endCoord = coord
        end
        table.insert(self.points, coord)
        self:apply()
        return true
    elseif touch.state == ENDED and self.state == TOOL_STATE_DRAG then
        self.state = TOOL_STATE_IDLE
        self:apply()
        saveSnapshot()
        return true
    end
    
    return false
end

function Tool:mirrorX(x)
    return (sx-1) - x
end

function Tool:mirrorY(y)
    return (sy-1) - y
end

function Tool:mirrorZ(z)
    return (sz-1) - z
end

function Tool:applyPoints(...)
    for k,v in pairs(self.points) do
        volume:set(v, ...)  
        if self.mirror.x then
            volume:set(self:mirrorX(v.x), v.y, v.z, ...)              
        end
        if self.mirror.y then
            volume:set(v.x, self:mirrorY(v.y), v.z, ...)              
        end
        if self.mirror.z then
            volume:set(v.x, v.y, self:mirrorZ(v.z), ...)              
        end
    end    
end

function Tool:applyBox(...)
    local minX = math.min(self.startCoord.x, self.endCoord.x)
    local maxX = math.max(self.startCoord.x, self.endCoord.x)
    local minY = math.min(self.startCoord.y, self.endCoord.y)
    local maxY = math.max(self.startCoord.y, self.endCoord.y)
    local minZ = math.min(self.startCoord.z, self.endCoord.z)
    local maxZ = math.max(self.startCoord.z, self.endCoord.z)
    
    for x = minX, maxX do
        for y = minY, maxY do
            for z = minZ, maxZ do
                if self.toolMode == TOOL_REPLACE then
                    if volume:get(x, y, z, BLOCK_ID) ~= 0 then
                        volume:set(x, y, z, ...)                                           
                    end
                elseif self.toolMode == TOOL_ADD then
                    if volume:get(x, y, z, BLOCK_ID) == 0 then
                        volume:set(x, y, z, ...) 
                        -- print("Current axis:", vec3(x,y,z))
                    end                                          
                else
                    volume:set(x, y, z, ...)                   
                end                
            end
        end
    end
end

function Tool:applyLine(...)
    if self.endCoord == self.startCoord then
        volume:set(self.startCoord, ...)
        return
    end
    
    local dir = (self.endCoord-self.startCoord)
    local args = {...}
    volume:raycast(self.startCoord + vec3(0.5, 0.5, 0.5), dir:normalize(), dir:len(), function(coord, id, face) 
        if coord then
            if self.toolMode == TOOL_REPLACE then
                if volume:get(coord, BLOCK_ID) ~= 0 then
                    volume:set(coord, table.unpack(args))
                end
            elseif self.toolMode == TOOL_ADD then
                if volume:get(coord, BLOCK_ID) == 0 then
                    volume:set(coord, table.unpack(args))
                end
            else
                volume:set(coord, table.unpack(args))
            end
            return false
        else
            return true
        end
    end)    
end

function Tool:apply()
    if self.toolMode == TOOL_ADD or self.toolMode == TOOL_REPLACE then
        if self.toolType == TOOL_TYPE_POINT then
            self:applyPoints("name", self.toolUnitType, "color", self.toolColor)
        elseif self.toolType == TOOL_TYPE_BOX then
            self:applyBox("name", self.toolUnitType, "color", self.toolColor)
        elseif self.toolType == TOOL_TYPE_LINE then
            self:applyLine("name", self.toolUnitType, "color", self.toolColor)
        end
    elseif self.toolMode == TOOL_ERASE then
        if self.toolType == TOOL_TYPE_POINT then
            self:applyPoints(0)
        elseif self.toolType == TOOL_TYPE_BOX then
            self:applyBox(0)    
        elseif self.toolType == TOOL_TYPE_LINE then
            self:applyLine(0)                    
        end
    elseif self.toolMode == TOOL_GET then
        local s = volume:get(self.startCoord, BLOCK_STATE)
        local n = volume:get(self.startCoord, BLOCK_NAME)
        if s then
            local r = (s>>24) & 255
            local g = (s>>16) & 255   
            local b = (s>>8) & 255     
            self.toolColor = color(r,g,b)
            Color = self.toolColor
        else
            self.toolColor = color(255)
            Color = self.toolColor
        end
        if n then self.toolUnitType = n end
    end
end

----

