-- BlockImventory
-- A basic inventory class for viewing and selecting block types.

Inventory = class()

function Inventory:init(rows, cols, gridSize, border, spacing, itemCallback)
    self.gridRows = rows or 8
    self.gridCols = cols or 8
    self.gridSize = gridSize or 64
    self.border = border or 5
    self.spacing = spacing or 5
    self.slots = {}
    
    -- 面板的长度，宽度
    local w,h =
    self.gridCols * self.gridSize + self.border * 2 + (self.gridCols-1) * self.spacing,
    self.gridRows * self.gridSize + self.border * 2 + (self.gridRows-1) * self.spacing
    
    -- 新建一个 panel
    self.panel = ui.panel
    {
        anchor = vec2(0.5, 1.0),
        pivot = vec2(0.5, 0.0),
        w = w,
        h = h,
        align = {h = ui.CENTER, v = ui.TOP},
        -- bg = "UI:Grey Panel",
        bg = asset.builtin.UI.Grey_Panel,
        fill = color(0, 19),
        inset = 10,
    }
    
    -- 把 panel 设置为圆角矩形
    self.panel.cornerRadius = 5
    -- 整个面板的标题
    self.title = ui.label
    {
        text = "",
        fontSize = 30,
        w = 200,
        h = 25,
        pivot = vec2(0.5, 0),
        anchor = vec2(0.5, 1.0),
        align = {h = ui.CENTER, v = ui.TOP}
    }
    self.panel:addChild(self.title)
    
    -- 选中的项目的标题位置
    self.title.anchor = vec2(0.5, 1.0)
    
    self.itemCallback = itemCallback
    
    -- 创建所有 slot 槽位：slot: { index=1, button={button,onPressed}, }
    for y = 1,self.gridRows do
        for x = 1,self.gridCols do
            local button = self:slotButton(x, self.gridRows+1-y)
            local slot = {button = button}
            -- 生成每个 slot ，#self.slots 的值从 0 开始递增
            slot.index = #self.slots+1            
            slot.button.onPressed = function(b)
                if self.itemCallback then
                    -- 在这里向回调函数传入参数：invertory，slot
                    self.itemCallback(self, slot)
                end
            end
            -- 最终 slot 表的结构为： { index=1, button = button}
            -- 让每个槽位的按钮的文字标签在底部显示
            -- slot.button.label.anchor = vec2(0.5, 0.1)
            slot.button.label.anchor = device.ModelItemInvLabel.show
            slot.button.label.fontSize = 12
            -- 把生成的槽位项目 slot 插入到槽位表中 self.slots
            table.insert(self.slots, slot)
        end
    end
end

-- 保存当前槽位的项目的选择状态
function Inventory:save(name)
    -- 表数据结构 data: { selected = 1, slots = { slot, slot, slot } }
    -- slot: { block = "block name"}
    -- {"selected":7,"slots":[{"block":"Grass"},{"block":"Dirt"},{"block":"Stone"},{"block":"Redstone Ore"},{"block":"Planks"},{"block":"Wood"},{"block":"Water"},[]]}
    -- {"selected":2,"slots":[[],{"block":"myNewHouse.cvox"},[],[],[],[],[],[]]}
    local data = {}
    data.selected = self.selected and self.selected.index
    data.slots = {}
    for k,v in pairs(self.slots) do
        local slot = {}
        if v.block then
            slot.block = v.block.name
        end
        table.insert(data.slots, slot)
    end
    -- print("Inventory:save(name), data:", json.encode(data))
    saveText(name, json.encode(data) )
end

function Inventory:load(name)
    local txt = readText(name)
    if txt then
        local data = json.decode(txt)
        local selected = data.selected
        -- 遍历 inventory 的所有槽位
        for k,v in pairs(self.slots) do
            local blockName = data.slots[v.index].block
            if blockName then
                -- 获取 block 名字
                self:setItem(v.index, scene.voxels.blocks:get(blockName))
                if v.index == selected then
                    self:setSelected(v)
                end
            end
        end
    end
end

function Inventory:slide(anchorX, anchorY)
    local x,y = anchorX, anchorY
    -- tween(0.6, self.panel.anchor, {x = x or 0.5, y = y or 1.0}, tween.easing.cubicInOut)
    tween(0.6, self.panel.anchor, {y = y or 1.0}, tween.easing.cubicInOut)
end

--[[
function Inventory:close(anchorX, anchorY)
    local x,y = anchorX, anchorY    
    tween(0.6, self.panel.anchor, {y = y or 0.1}, tween.easing.cubicInOut) 
end
--]]

-- 返回某个槽位的对象(按钮)
function Inventory:getSlot(i)
    return self.slots[i]
end

-- 返回总槽位数
function Inventory:slotCount()
    return #self.slots
end

function Inventory:setSelected(slot)
    if self.selected then
        self.selected.button.selected = false
    end
    -- 选中的槽位
    self.selected = slot
    if self.selected then
        self.selected.button.selected = true
    end
end

function Inventory:getSelected()
    return self.selected
end


-- Add item to next empty slot
function Inventory:addItem(block)
    for k,v in pairs(self.slots) do
        if v.block == nil then
            self:setItem(v.index, block)
            return true
        end
    end
    return false
end

-- 清除整个 Inventory 面板上所有槽位中的数据
function Inventory:removeItem(name)
    local bName = name or ""
    for k,v in pairs(self.slots) do        
        local slot = v
        slot.button.icon.img = nil
        slot.button.label.text = bName
        slot.block = nil
    end
end

-- Set item for a specific slot
-- 设置每个槽位的图标和文字：考虑把 model 和 block 的处理合并在一起
function Inventory:setItem(index, block)
    local slot = self.slots[index]
    
    if block then
        slot.block = block
        if block.hasIcon then
            -- 处理 block 方块类型的流程
            slot.button.icon.img = block.icon
            slot.button.label.text = ""
        else
            -- 处理 volume model 类型的流程
            slot.button.icon.img = block.icon
            slot.button.label.text = block.name or block.longName
        end
    end    
end

-- Create a new slot button
function Inventory:slotButton(x,y)
    -- 这里的每个按钮都比较独立，跟其他按钮没有交互作用
    local button = ui.button
    {
        x = self.border + self.gridSize * (x-1) + self.spacing * (x-1),
        y = self.border + self.gridSize * (y-1) + self.spacing * (y-1),
        w = self.gridSize,
        h = self.gridSize,
        normalBg = readImage(asset.builtin.UI.Green_Panel),
        normalFill = color(53, 47),
        selectedBg = readImage(asset.builtin.UI.Blue_Panel),
        selectedFill = color(255, 255, 255, 255)
        --selectedBg = readImage()
        --bg = "Documents:grey_button11",
        --normalFill = color(79, 79, 79, 255)
    }
    
    -- 设定按钮图标
    button.icon = ui.image({x = 5, y = 5, w = self.gridSize-10, h = self.gridSize-10})
    -- 将图标加入按钮的子节点中
    button:addChild(button.icon)
    button.unselectedFill = color(255, 255, 255, 255)
    button.selectedFill = color(34, 219, 241)
    button.opaque = false
    
    -- 把每个新建的按钮都加入 self.panel 的子节点，统一更新，绘制
    self.panel:addChild(button)
    
    return button
end

function Inventory:draw()
    -- panel 更新与绘制
    self.panel:update()
    self.panel:draw()
end

----

