local WorkerManager = require('workers')
local webview = require('webview')
local base64 = require("base64")

local b64 = base64.b64

-- Class declarations
local LuaParallelInstance = class()

local POOL_SIZE_INITIAL = 4
local POOL_SIZE_MINIMUM = 2

-- Create JS environment & channel manager
local wv = webview()
local wm = WorkerManager(6767, wv)

local workerPool = {}
local function initThreadPool(num)
	for i=1,num do
		-- Add new worker to pool
		local worker = wm:newWorker()
		worker:send({
			wasmURL = "file://" .. asset._blobs.Parallel.web.libs.wasmoon.glue.path
		});
		table.insert(workerPool, worker)
	end
end
initThreadPool(POOL_SIZE_INITIAL)

local function getWorker()
	-- Get from workerPool
	local w = table.remove(workerPool)
	
	-- Add new worker to the pool
	-- TODO: This needs to be nonblocking
	if #workerPool < POOL_SIZE_MINIMUM then
		local worker = wm:newWorker()
		worker:send({
			wasmURL = "file://" .. asset._blobs.Parallel.web.libs.wasmoon.glue.path
		});
		table.insert(workerPool, worker)
	end
	
	return w
end

function LuaParallelInstance:init(source)
	-- Get a worker instance
	self._worker = getWorker()
	
	if type(source) == "string" then
		local workerSource = [[
			// Basic output functions
			L.global.set("print", print);
			L.global.set("error", error);
			
			// Load the Lua code
			const code = await codea.recv();
			try {
				await L.doString(code);
			} catch(e) {
				error(e.message);
			}
			
			// Signal to Codea that the instance is ready
			codea.send("_ready_");
			
			while (true) {
				const invocation = await codea.recv();
				
				let start = performance.now();
				const result = L.global.call(invocation.fn, ...invocation.args);
				let end = performance.now();
				//print("Lua call took: " + (end - start) + " ms");
				if ((typeof result) === "object" && result.constructor.name == "Promise") {
					result.then((val)=>{
						codea.send({
							data: val,
							id: invocation.id
						});
					})
				}
				else
				{
					codea.send({
						data: result,
						id: invocation.id
					});
				}
			}
		]]
		
		-- Send the Lua worker code
		self._worker:send({
			code = workerSource,
			args = {}
		});
		
		-- Send the Lua instance code
		self._worker:send(source);
		
	elseif type(source) == "table" then
		
		-- Starting with the passed function, move up the upvalue chain to ensure all upvalues are accounted for.
		local function transferableDump(fn, D)
			local D = D or {}
			
			-- Declare
			local dumpValue
			local dumpFunction
			
			dumpValue = function(val, id)
				local typ = type(val)
				if typ == "function" then
					D[id] = dumpFunction(val, id)
				else
					D[id] = { data = val }
				end
			end
			
			dumpFunction = function(fn, id)
				-- Dump our function
				local f = {
					fn = true,
					uvals = {},
					data = string.dump(fn)
				}
				D[id] = f
				
				-- Dump all the upvalues too
				local info = debug.getinfo(fn, "u")
				if info.nups > 0 then
					for up=1,info.nups do
						local name, value = debug.getupvalue(fn, up)
						
						-- Ignore the _ENV upvalue as this will be initialised
						-- when we load the chunk in wasmoon.
						if name == "_ENV" then
							f.uvals[up] = "_ENV"
						else
							local id = tostring(debug.upvalueid(fn, up)):sub(11)
							
							-- Add to function upvalues
							f.uvals[up] = id
							
							-- Add to value map if multiple functions are using the same
							-- upvalue.
							if D[id] == nil then
								dumpValue(value, id)
							end
						end
					end
				end
				
				return f
			end
			
			-- The function to execute will be the first value
			local key = tostring(fn):sub(11)
			dumpFunction(fn, key)
			
			-- Return the function key
			return key
		end
		
		local D = {
			fnMap = {}
		}
		for fnName, fn in pairs(source) do
			D.fnMap[fnName] = transferableDump(fn, D)
		end
		
		local workerSource = [[
			// Add _LoadDump function
			await L.doString(`
				function __LoadDump(D)	
					local clone
					clone = function(v)
						if type(v) == "userdata" then
							local nt = {}
							for k, v in pairs(v) do
								nt[k] = clone(v)
							end
							for i, v in ipairs(v) do
								nt[i] = clone(v)
							end
							return nt
						else
							return v
						end
					end
					
					-- Copy everything to local scope
					local D = clone(D)
					
					-- Move the fnMap
					local fnMap = D.fnMap
					D.fnMap = nil
					
					-- Load all functions
					for k, v in pairs(D) do
						if v.fn then
							v.data = load(v.data)
						end
					end
					
					-- Set function upvalues
					for k, v in pairs(D) do
						if v.fn then
							for uv,key in ipairs(v.uvals) do
								if key == "_ENV" then
									debug.setupvalue(v.data, uv, _ENV)
								else
									debug.setupvalue(v.data, uv, D[key].data)
								end
							end
						end
					end
					
					-- Load all into _ENV
					for k, id in pairs(fnMap) do
						_ENV[k] = function(...)
							return select(2, xpcall(D[id].data, function(m)
								error(m)
							end, ...))
						end
					end
				end
			`);
			
			// Load the Lua dump structure
			const dump = await codea.recv();
			L.global.call("__LoadDump", dump);
			
			// Signal to Codea that the instance is ready
			codea.send("_ready_");
			
			while (true) {
				const invocation = await codea.recv();
				
				let result;
				try {
					result = L.global.call(invocation.fn, ...invocation.args);
				} catch(e) {
					error(e.message);
				}
				if ((typeof result) === "object" && result.constructor.name == "Promise") {
					result.then((val)=>{
						codea.send({
							data: val,
							id: invocation.id
						});
					})
				}
				else
				{
					codea.send({
						data: result,
						id: invocation.id
					});
				}
			}
		]]
		
		-- Send the Lua worker code
		self._worker:send({
			code = workerSource,
			args = {}
		});
		
		-- Send the dump table
		self._worker:send(D);
	end
	
	-- Wait for 'ready' signal
	assert(self._worker:recv() == "_ready_")
	
	-- Each call is assigned an ID so we can separate results
	local callId = 1
	
	setmetatable(self, {
		__index = function(t, k)
			rawset(t, k, function(...)
				self._worker:send({
					fn = k,
					args = {...},
					id = callId
				});
				
				-- Create a new receiver object for this call
				local receiver = self._worker:newReceiver(callId)
				
				-- Increment callId
				callId = callId + 1
				
				return receiver
			end)
			return t[k]
		end
	})
end

local exports = setmetatable({}, {
	__call = function(t, source)
		return LuaParallelInstance(source)
	end
})

-- Add automated update function
exports.updatePerFrameLimit = 32
local _tweenup = tween.update
tween.update = function(...)
	_tweenup(...)
	wm:update(limit or exports.updatePerFrameLimit)
end

-- Set global
return {
	parallel = exports
}