(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.wasmoon = {}));
})(this, (function (exports) { 'use strict';

    exports.LuaReturn = void 0;
    (function (LuaReturn) {
        LuaReturn[LuaReturn["Ok"] = 0] = "Ok";
        LuaReturn[LuaReturn["Yield"] = 1] = "Yield";
        LuaReturn[LuaReturn["ErrorRun"] = 2] = "ErrorRun";
        LuaReturn[LuaReturn["ErrorSyntax"] = 3] = "ErrorSyntax";
        LuaReturn[LuaReturn["ErrorMem"] = 4] = "ErrorMem";
        LuaReturn[LuaReturn["ErrorErr"] = 5] = "ErrorErr";
        LuaReturn[LuaReturn["ErrorFile"] = 6] = "ErrorFile";
    })(exports.LuaReturn || (exports.LuaReturn = {}));
    const PointerSize = 4;
    const LUA_MULTRET = -1;
    const LUAI_MAXSTACK = 1000000;
    const LUA_REGISTRYINDEX = -LUAI_MAXSTACK - 1000;
    exports.LuaType = void 0;
    (function (LuaType) {
        LuaType[LuaType["None"] = -1] = "None";
        LuaType[LuaType["Nil"] = 0] = "Nil";
        LuaType[LuaType["Boolean"] = 1] = "Boolean";
        LuaType[LuaType["LightUserdata"] = 2] = "LightUserdata";
        LuaType[LuaType["Number"] = 3] = "Number";
        LuaType[LuaType["String"] = 4] = "String";
        LuaType[LuaType["Table"] = 5] = "Table";
        LuaType[LuaType["Function"] = 6] = "Function";
        LuaType[LuaType["Userdata"] = 7] = "Userdata";
        LuaType[LuaType["Thread"] = 8] = "Thread";
    })(exports.LuaType || (exports.LuaType = {}));
    exports.LuaEventCodes = void 0;
    (function (LuaEventCodes) {
        LuaEventCodes[LuaEventCodes["Call"] = 0] = "Call";
        LuaEventCodes[LuaEventCodes["Ret"] = 1] = "Ret";
        LuaEventCodes[LuaEventCodes["Line"] = 2] = "Line";
        LuaEventCodes[LuaEventCodes["Count"] = 3] = "Count";
        LuaEventCodes[LuaEventCodes["TailCall"] = 4] = "TailCall";
    })(exports.LuaEventCodes || (exports.LuaEventCodes = {}));
    exports.LuaEventMasks = void 0;
    (function (LuaEventMasks) {
        LuaEventMasks[LuaEventMasks["Call"] = 1] = "Call";
        LuaEventMasks[LuaEventMasks["Ret"] = 2] = "Ret";
        LuaEventMasks[LuaEventMasks["Line"] = 4] = "Line";
        LuaEventMasks[LuaEventMasks["Count"] = 8] = "Count";
    })(exports.LuaEventMasks || (exports.LuaEventMasks = {}));
    exports.LuaLibraries = void 0;
    (function (LuaLibraries) {
        LuaLibraries["Base"] = "_G";
        LuaLibraries["Coroutine"] = "coroutine";
        LuaLibraries["Table"] = "table";
        LuaLibraries["IO"] = "io";
        LuaLibraries["OS"] = "os";
        LuaLibraries["String"] = "string";
        LuaLibraries["UTF8"] = "utf8";
        LuaLibraries["Math"] = "math";
        LuaLibraries["Debug"] = "debug";
        LuaLibraries["Package"] = "package";
    })(exports.LuaLibraries || (exports.LuaLibraries = {}));
    class LuaTimeoutError extends Error {
    }

    class Decoration {
        constructor(target, options) {
            this.target = target;
            this.options = options;
        }
    }
    function decorate(target, options) {
        return new Decoration(target, options);
    }

    class Pointer extends Number {
    }

    class MultiReturn extends Array {
    }

    const INSTRUCTION_HOOK_COUNT = 1000;
    class Thread {
        constructor(lua, typeExtensions, address, parent) {
            this.closed = false;
            this.lua = lua;
            this.typeExtensions = typeExtensions;
            this.address = address;
            this.parent = parent;
        }
        newThread() {
            const address = this.lua.lua_newthread(this.address);
            if (!address) {
                throw new Error('lua_newthread returned a null pointer');
            }
            return new Thread(this.lua, this.typeExtensions, address);
        }
        resetThread() {
            this.assertOk(this.lua.lua_resetthread(this.address));
        }
        loadString(luaCode, name) {
            const size = this.lua.module.lengthBytesUTF8(luaCode);
            const pointerSize = size + 1;
            const bufferPointer = this.lua.module._malloc(pointerSize);
            try {
                this.lua.module.stringToUTF8(luaCode, bufferPointer, pointerSize);
                this.assertOk(this.lua.luaL_loadbufferx(this.address, bufferPointer, size, name !== null && name !== void 0 ? name : bufferPointer, null));
            }
            finally {
                this.lua.module._free(bufferPointer);
            }
        }
        loadFile(filename) {
            this.assertOk(this.lua.luaL_loadfilex(this.address, filename, null));
        }
        loadBytecode(bytecode, name) {
            const bytecodeBuf = new Uint8Array(bytecode.split("").map(function (c) {
                return c.charCodeAt(0);
            }));
            const size = bytecodeBuf.length;
            const bufferPointer = this.lua.module._malloc(size);
            try {
                this.lua.module.HEAPU8.set(bytecodeBuf, bufferPointer);
                this.assertOk(this.lua.luaL_loadbufferx(this.address, bufferPointer, size, name !== null && name !== void 0 ? name : bufferPointer, null));
            }
            finally {
                this.lua.module._free(bufferPointer);
            }
        }
        resume(argCount = 0) {
            const dataPointer = this.lua.module._malloc(PointerSize);
            try {
                this.lua.module.setValue(dataPointer, 0, 'i32');
                const luaResult = this.lua.lua_resume(this.address, null, argCount, dataPointer);
                return {
                    result: luaResult,
                    resultCount: this.lua.module.getValue(dataPointer, 'i32'),
                };
            }
            finally {
                this.lua.module._free(dataPointer);
            }
        }
        getTop() {
            return this.lua.lua_gettop(this.address);
        }
        setTop(index) {
            this.lua.lua_settop(this.address, index);
        }
        remove(index) {
            return this.lua.lua_remove(this.address, index);
        }
        setField(index, name, value) {
            index = this.lua.lua_absindex(this.address, index);
            this.pushValue(value);
            this.lua.lua_setfield(this.address, index, name);
        }
        async run(argCount = 0, options) {
            const originalTimeout = this.timeout;
            try {
                if ((options === null || options === void 0 ? void 0 : options.timeout) !== undefined) {
                    this.setTimeout(Date.now() + options.timeout);
                }
                let resumeResult = this.resume(argCount);
                while (resumeResult.result === exports.LuaReturn.Yield) {
                    if (this.timeout && Date.now() > this.timeout) {
                        if (resumeResult.resultCount > 0) {
                            this.pop(resumeResult.resultCount);
                        }
                        throw new LuaTimeoutError(`thread timeout exceeded`);
                    }
                    if (resumeResult.resultCount > 0) {
                        const lastValue = this.getValue(-1);
                        this.pop(resumeResult.resultCount);
                        if (lastValue === Promise.resolve(lastValue)) {
                            await lastValue;
                        }
                        else {
                            await new Promise((resolve) => setImmediate(resolve));
                        }
                    }
                    else {
                        await new Promise((resolve) => setImmediate(resolve));
                    }
                    resumeResult = this.resume(0);
                }
                this.assertOk(resumeResult.result);
                return this.getStackValues();
            }
            finally {
                if ((options === null || options === void 0 ? void 0 : options.timeout) !== undefined) {
                    this.setTimeout(originalTimeout);
                }
            }
        }
        runSync(argCount = 0) {
            const base = this.getTop() - argCount - 1;
            this.assertOk(this.lua.lua_pcallk(this.address, argCount, LUA_MULTRET, 0, 0, null));
            return this.getStackValues(base);
        }
        pop(count = 1) {
            this.lua.lua_pop(this.address, count);
        }
        call(name, ...args) {
            const type = this.lua.lua_getglobal(this.address, name);
            if (type !== exports.LuaType.Function) {
                throw new Error(`A function of type '${type}' was pushed, expected is ${exports.LuaType.Function}`);
            }
            for (const arg of args) {
                this.pushValue(arg);
            }
            const base = this.getTop() - args.length - 1;
            this.lua.lua_callk(this.address, args.length, LUA_MULTRET, 0, null);
            return this.getStackValues(base);
        }
        getStackValues(start = 0) {
            const returns = this.getTop() - start;
            const returnValues = new MultiReturn(returns);
            for (let i = 0; i < returns; i++) {
                returnValues[i] = this.getValue(start + i + 1, undefined, undefined, true);
            }
            return returnValues;
        }
        stateToThread(L) {
            var _a;
            return L === ((_a = this.parent) === null || _a === void 0 ? void 0 : _a.address) ? this.parent : new Thread(this.lua, this.typeExtensions, L, this.parent || this);
        }
        pushValue(rawValue, userdata) {
            var _a;
            const decoratedValue = this.getValueDecorations(rawValue);
            const target = (_a = decoratedValue.target) !== null && _a !== void 0 ? _a : undefined;
            if (target instanceof Thread) {
                const isMain = this.lua.lua_pushthread(target.address) === 1;
                if (!isMain) {
                    this.lua.lua_xmove(target.address, this.address, 1);
                }
                return;
            }
            const startTop = this.getTop();
            switch (typeof target) {
                case 'undefined':
                    this.lua.lua_pushnil(this.address);
                    break;
                case 'number':
                    if (Number.isInteger(target)) {
                        this.lua.lua_pushinteger(this.address, target);
                    }
                    else {
                        this.lua.lua_pushnumber(this.address, target);
                    }
                    break;
                case 'string':
                    this.lua.lua_pushbstring(this.address, target);
                    break;
                case 'boolean':
                    this.lua.lua_pushboolean(this.address, target ? 1 : 0);
                    break;
                default:
                    if (!this.typeExtensions.find((wrapper) => wrapper.extension.pushValue(this, decoratedValue, userdata))) {
                        throw new Error(`The type '${typeof target}' is not supported by Lua`);
                    }
            }
            if (decoratedValue.options.metatable) {
                this.setMetatable(-1, decoratedValue.options.metatable);
            }
            if (this.getTop() !== startTop + 1) {
                throw new Error(`pushValue expected stack size ${startTop + 1}, got ${this.getTop()}`);
            }
        }
        setMetatable(index, metatable) {
            index = this.lua.lua_absindex(this.address, index);
            if (this.lua.lua_getmetatable(this.address, index)) {
                this.pop(1);
                const name = this.getMetatableName(index);
                throw new Error(`data already has associated metatable: ${name || 'unknown name'}`);
            }
            this.pushValue(metatable);
            this.lua.lua_setmetatable(this.address, index);
        }
        getMetatableName(index) {
            const metatableNameType = this.lua.luaL_getmetafield(this.address, index, '__name');
            if (metatableNameType === exports.LuaType.Nil) {
                return undefined;
            }
            if (metatableNameType !== exports.LuaType.String) {
                this.pop(1);
                return undefined;
            }
            const name = this.lua.lua_tolstring(this.address, -1, null);
            this.pop(1);
            return name;
        }
        getValue(index, inputType, userdata, stringsAsArrays) {
            index = this.lua.lua_absindex(this.address, index);
            const type = inputType !== null && inputType !== void 0 ? inputType : this.lua.lua_type(this.address, index);
            switch (type) {
                case exports.LuaType.None:
                    return undefined;
                case exports.LuaType.Nil:
                    return null;
                case exports.LuaType.Number:
                    return this.lua.lua_tonumberx(this.address, index, null);
                case exports.LuaType.String:
                    if (stringsAsArrays === true) {
                        return this.lua.lua_tou8array(this.address, index);
                    }
                    else {
                        return this.lua.lua_tolstring(this.address, index, null);
                    }
                case exports.LuaType.Boolean:
                    return Boolean(this.lua.lua_toboolean(this.address, index));
                case exports.LuaType.Thread:
                    return this.stateToThread(this.lua.lua_tothread(this.address, index));
                default: {
                    let metatableName;
                    if (type === exports.LuaType.Table || type === exports.LuaType.Userdata) {
                        metatableName = this.getMetatableName(index);
                    }
                    const typeExtensionWrapper = this.typeExtensions.find((wrapper) => wrapper.extension.isType(this, index, type, metatableName));
                    if (typeExtensionWrapper) {
                        return typeExtensionWrapper.extension.getValue(this, index, userdata);
                    }
                    console.warn(`The type '${this.lua.lua_typename(this.address, type)}' returned is not supported on JS`);
                    return new Pointer(this.lua.lua_topointer(this.address, index));
                }
            }
        }
        close() {
            if (this.isClosed()) {
                return;
            }
            if (this.hookFunctionPointer) {
                this.lua.module.removeFunction(this.hookFunctionPointer);
            }
            this.closed = true;
        }
        setTimeout(timeout) {
            if (timeout && timeout > 0) {
                if (!this.hookFunctionPointer) {
                    this.hookFunctionPointer = this.lua.module.addFunction(() => {
                        if (Date.now() > timeout) {
                            this.pushValue(new LuaTimeoutError(`thread timeout exceeded`));
                            this.lua.lua_error(this.address);
                        }
                    }, 'vii');
                }
                this.lua.lua_sethook(this.address, this.hookFunctionPointer, exports.LuaEventMasks.Count, INSTRUCTION_HOOK_COUNT);
                this.timeout = timeout;
            }
            else if (this.hookFunctionPointer) {
                this.hookFunctionPointer = undefined;
                this.timeout = undefined;
                this.lua.lua_sethook(this.address, null, 0, 0);
            }
        }
        getTimeout() {
            return this.timeout;
        }
        getPointer(index) {
            return new Pointer(this.lua.lua_topointer(this.address, index));
        }
        isClosed() {
            var _a;
            return !this.address || this.closed || Boolean((_a = this.parent) === null || _a === void 0 ? void 0 : _a.isClosed());
        }
        indexToString(index) {
            const str = this.lua.luaL_tolstring(this.address, index, null);
            this.pop();
            return str;
        }
        dumpStack(log = console.log) {
            const top = this.getTop();
            for (let i = 1; i <= top; i++) {
                const type = this.lua.lua_type(this.address, i);
                const typename = this.lua.lua_typename(this.address, type);
                const pointer = this.getPointer(i);
                const name = this.indexToString(i);
                const value = this.getValue(i, type);
                log(i, typename, pointer, name, value);
            }
        }
        assertOk(result) {
            if (result !== exports.LuaReturn.Ok && result !== exports.LuaReturn.Yield) {
                const resultString = exports.LuaReturn[result];
                const error = new Error(`Lua Error(${resultString}/${result})`);
                if (this.getTop() > 0) {
                    if (result === exports.LuaReturn.ErrorMem) {
                        error.message = this.lua.lua_tolstring(this.address, -1, null);
                    }
                    else {
                        const luaError = this.getValue(-1);
                        if (luaError instanceof Error) {
                            error.stack = luaError.stack;
                        }
                        error.message = this.indexToString(-1);
                    }
                }
                if (result !== exports.LuaReturn.ErrorMem) {
                    try {
                        this.lua.luaL_traceback(this.address, this.address, null, 1);
                        const traceback = this.lua.lua_tolstring(this.address, -1, null);
                        if (traceback.trim() !== 'stack traceback:') {
                            error.message = `${error.message}\n${traceback}`;
                        }
                        this.pop(1);
                    }
                    catch (err) {
                        console.warn('Failed to generate stack trace', err);
                    }
                }
                throw error;
            }
        }
        getValueDecorations(value) {
            return value instanceof Decoration ? value : new Decoration(value, {});
        }
    }

    class Global extends Thread {
        constructor(cmodule, shouldTraceAllocations) {
            if (shouldTraceAllocations) {
                const memoryStats = { memoryUsed: 0 };
                const allocatorFunctionPointer = cmodule.module.addFunction((_userData, pointer, oldSize, newSize) => {
                    if (newSize === 0) {
                        if (pointer) {
                            memoryStats.memoryUsed -= oldSize;
                            cmodule.module._free(pointer);
                        }
                        return 0;
                    }
                    const endMemoryDelta = pointer ? newSize - oldSize : newSize;
                    const endMemory = memoryStats.memoryUsed + endMemoryDelta;
                    if (newSize > oldSize && memoryStats.memoryMax && endMemory > memoryStats.memoryMax) {
                        return 0;
                    }
                    const reallocated = cmodule.module._realloc(pointer, newSize);
                    if (reallocated) {
                        memoryStats.memoryUsed = endMemory;
                    }
                    return reallocated;
                }, 'iiiii');
                super(cmodule, [], cmodule.lua_newstate(allocatorFunctionPointer, null));
                this.memoryStats = memoryStats;
                this.allocatorFunctionPointer = allocatorFunctionPointer;
            }
            else {
                super(cmodule, [], cmodule.luaL_newstate());
            }
            if (this.isClosed()) {
                throw new Error('Global state could not be created (probably due to lack of memory)');
            }
        }
        close() {
            if (this.isClosed()) {
                return;
            }
            super.close();
            this.lua.lua_close(this.address);
            if (this.allocatorFunctionPointer) {
                this.lua.module.removeFunction(this.allocatorFunctionPointer);
            }
            for (const wrapper of this.typeExtensions) {
                wrapper.extension.close();
            }
        }
        registerTypeExtension(priority, extension) {
            this.typeExtensions.push({ extension, priority });
            this.typeExtensions.sort((a, b) => b.priority - a.priority);
        }
        loadLibrary(library) {
            switch (library) {
                case exports.LuaLibraries.Base:
                    this.lua.luaopen_base(this.address);
                    break;
                case exports.LuaLibraries.Coroutine:
                    this.lua.luaopen_coroutine(this.address);
                    break;
                case exports.LuaLibraries.Table:
                    this.lua.luaopen_table(this.address);
                    break;
                case exports.LuaLibraries.IO:
                    this.lua.luaopen_io(this.address);
                    break;
                case exports.LuaLibraries.OS:
                    this.lua.luaopen_os(this.address);
                    break;
                case exports.LuaLibraries.String:
                    this.lua.luaopen_string(this.address);
                    break;
                case exports.LuaLibraries.UTF8:
                    this.lua.luaopen_string(this.address);
                    break;
                case exports.LuaLibraries.Math:
                    this.lua.luaopen_math(this.address);
                    break;
                case exports.LuaLibraries.Debug:
                    this.lua.luaopen_debug(this.address);
                    break;
                case exports.LuaLibraries.Package:
                    this.lua.luaopen_package(this.address);
                    break;
            }
            this.lua.lua_setglobal(this.address, library);
        }
        get(name) {
            const type = this.lua.lua_getglobal(this.address, name);
            const value = this.getValue(-1, type);
            this.pop();
            return value;
        }
        set(name, value) {
            this.pushValue(value);
            this.lua.lua_setglobal(this.address, name);
        }
        getTable(name, callback) {
            const startStackTop = this.getTop();
            const type = this.lua.lua_getglobal(this.address, name);
            try {
                if (type !== exports.LuaType.Table) {
                    throw new TypeError(`Unexpected type in ${name}. Expected ${exports.LuaType[exports.LuaType.Table]}. Got ${exports.LuaType[type]}.`);
                }
                callback(startStackTop + 1);
            }
            finally {
                if (this.getTop() !== startStackTop + 1) {
                    console.warn(`getTable: expected stack size ${startStackTop} got ${this.getTop()}`);
                }
                this.setTop(startStackTop);
            }
        }
        getMemoryUsed() {
            return this.getMemoryStatsRef().memoryUsed;
        }
        getMemoryMax() {
            return this.getMemoryStatsRef().memoryMax;
        }
        setMemoryMax(max) {
            this.getMemoryStatsRef().memoryMax = max;
        }
        getMemoryStatsRef() {
            if (!this.memoryStats) {
                throw new Error('Memory allocations is not being traced, please build engine with { traceAllocations: true }');
            }
            return this.memoryStats;
        }
    }

    class LuaTypeExtension {
        constructor(thread, name) {
            this.thread = thread;
            this.name = name;
        }
        isType(_thread, _index, type, name) {
            return type === exports.LuaType.Userdata && name === this.name;
        }
        getValue(thread, index, _userdata) {
            const refUserdata = thread.lua.luaL_testudata(thread.address, index, this.name);
            if (!refUserdata) {
                throw new Error(`data does not have the expected metatable: ${this.name}`);
            }
            const referencePointer = thread.lua.module.getValue(refUserdata, '*');
            return thread.lua.getRef(referencePointer);
        }
        pushValue(thread, decoratedValue, _userdata) {
            const { target } = decoratedValue;
            const pointer = thread.lua.ref(target);
            const userDataPointer = thread.lua.lua_newuserdatauv(thread.address, PointerSize, 0);
            thread.lua.module.setValue(userDataPointer, pointer, '*');
            if (exports.LuaType.Nil === thread.lua.luaL_getmetatable(thread.address, this.name)) {
                thread.pop(2);
                throw new Error(`metatable not found: ${this.name}`);
            }
            thread.lua.lua_setmetatable(thread.address, -2);
            return true;
        }
    }

    class ErrorTypeExtension extends LuaTypeExtension {
        constructor(thread, injectObject) {
            super(thread, 'js_error');
            this.gcPointer = thread.lua.module.addFunction((functionStateAddress) => {
                const userDataPointer = thread.lua.luaL_checkudata(functionStateAddress, 1, this.name);
                const referencePointer = thread.lua.module.getValue(userDataPointer, '*');
                thread.lua.unref(referencePointer);
                return exports.LuaReturn.Ok;
            }, 'ii');
            if (thread.lua.luaL_newmetatable(thread.address, this.name)) {
                const metatableIndex = thread.lua.lua_gettop(thread.address);
                thread.lua.lua_pushstring(thread.address, 'protected metatable');
                thread.lua.lua_setfield(thread.address, metatableIndex, '__metatable');
                thread.lua.lua_pushcclosure(thread.address, this.gcPointer, 0);
                thread.lua.lua_setfield(thread.address, metatableIndex, '__gc');
                thread.pushValue((jsRefError, key) => {
                    if (key === 'message') {
                        return jsRefError.message;
                    }
                    return null;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__index');
                thread.pushValue((jsRefError) => {
                    return jsRefError.message;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__tostring');
            }
            thread.lua.lua_pop(thread.address, 1);
            if (injectObject) {
                thread.set('Error', {
                    create: (message) => {
                        if (message && typeof message !== 'string') {
                            throw new Error('message must be a string');
                        }
                        return new Error(message);
                    },
                });
            }
        }
        pushValue(thread, decoration) {
            if (!(decoration.target instanceof Error)) {
                return false;
            }
            return super.pushValue(thread, decoration);
        }
        close() {
            this.thread.lua.module.removeFunction(this.gcPointer);
        }
    }
    function createTypeExtension$5(thread, injectObject) {
        return new ErrorTypeExtension(thread, injectObject);
    }

    class RawResult {
        constructor(count) {
            this.count = count;
        }
    }

    function decorateFunction(target, options) {
        return new Decoration(target, options);
    }
    class FunctionTypeExtension extends LuaTypeExtension {
        constructor(thread) {
            super(thread, 'js_function');
            this.functionRegistry = typeof FinalizationRegistry !== 'undefined'
                ? new FinalizationRegistry((func) => {
                    if (!this.thread.isClosed()) {
                        this.thread.lua.luaL_unref(this.thread.address, LUA_REGISTRYINDEX, func);
                    }
                })
                : undefined;
            if (!this.functionRegistry) {
                console.warn('FunctionTypeExtension: FinalizationRegistry not found. Memory leaks likely.');
            }
            this.gcPointer = thread.lua.module.addFunction((calledL) => {
                thread.lua.luaL_checkudata(calledL, 1, this.name);
                const userDataPointer = thread.lua.luaL_checkudata(calledL, 1, this.name);
                const referencePointer = thread.lua.module.getValue(userDataPointer, '*');
                thread.lua.unref(referencePointer);
                return exports.LuaReturn.Ok;
            }, 'ii');
            if (thread.lua.luaL_newmetatable(thread.address, this.name)) {
                thread.lua.lua_pushstring(thread.address, '__gc');
                thread.lua.lua_pushcclosure(thread.address, this.gcPointer, 0);
                thread.lua.lua_settable(thread.address, -3);
                thread.lua.lua_pushstring(thread.address, '__metatable');
                thread.lua.lua_pushstring(thread.address, 'protected metatable');
                thread.lua.lua_settable(thread.address, -3);
            }
            thread.lua.lua_pop(thread.address, 1);
            this.functionWrapper = thread.lua.module.addFunction((calledL) => {
                const calledThread = thread.stateToThread(calledL);
                const refUserdata = thread.lua.luaL_checkudata(calledL, thread.lua.lua_upvalueindex(1), this.name);
                const refPointer = thread.lua.module.getValue(refUserdata, '*');
                const { target, options } = thread.lua.getRef(refPointer);
                const argsQuantity = calledThread.getTop();
                const args = [];
                if (options.receiveThread) {
                    args.push(calledThread);
                }
                if (options.receiveArgsQuantity) {
                    args.push(argsQuantity);
                }
                else {
                    for (let i = 1; i <= argsQuantity; i++) {
                        const value = calledThread.getValue(i);
                        if (i !== 1 || !(options === null || options === void 0 ? void 0 : options.self) || value !== options.self) {
                            args.push(value);
                        }
                    }
                }
                try {
                    const result = target.apply(options === null || options === void 0 ? void 0 : options.self, args);
                    if (result === undefined) {
                        return 0;
                    }
                    else if (result instanceof RawResult) {
                        return result.count;
                    }
                    else if (result instanceof MultiReturn) {
                        for (const item of result) {
                            calledThread.pushValue(item);
                        }
                        return result.length;
                    }
                    else {
                        calledThread.pushValue(result);
                        return 1;
                    }
                }
                catch (err) {
                    if (err === Infinity) {
                        throw err;
                    }
                    calledThread.pushValue(err);
                    return calledThread.lua.lua_error(calledThread.address);
                }
            }, 'ii');
        }
        close() {
            this.thread.lua.module.removeFunction(this.gcPointer);
            this.thread.lua.module.removeFunction(this.functionWrapper);
        }
        isType(_thread, _index, type) {
            return type === exports.LuaType.Function;
        }
        pushValue(thread, decoration) {
            if (typeof decoration.target !== 'function') {
                return false;
            }
            const pointer = thread.lua.ref(decoration);
            const userDataPointer = thread.lua.lua_newuserdatauv(thread.address, PointerSize, 0);
            thread.lua.module.setValue(userDataPointer, pointer, '*');
            if (exports.LuaType.Nil === thread.lua.luaL_getmetatable(thread.address, this.name)) {
                thread.pop(1);
                thread.lua.unref(pointer);
                throw new Error(`metatable not found: ${this.name}`);
            }
            thread.lua.lua_setmetatable(thread.address, -2);
            thread.lua.lua_pushcclosure(thread.address, this.functionWrapper, 1);
            return true;
        }
        getValue(thread, index) {
            var _a;
            thread.lua.lua_pushvalue(thread.address, index);
            const func = thread.lua.luaL_ref(thread.address, LUA_REGISTRYINDEX);
            const jsFunc = (...args) => {
                if (thread.isClosed()) {
                    console.warn('Tried to call a function after closing lua state');
                    return;
                }
                const internalType = thread.lua.lua_rawgeti(thread.address, LUA_REGISTRYINDEX, func);
                if (internalType !== exports.LuaType.Function) {
                    const callMetafieldType = thread.lua.luaL_getmetafield(thread.address, -1, '__call');
                    thread.pop();
                    if (callMetafieldType !== exports.LuaType.Function) {
                        throw new Error(`A value of type '${internalType}' was pushed but it is not callable`);
                    }
                }
                for (const arg of args) {
                    thread.pushValue(arg);
                }
                const status = thread.lua.lua_pcallk(thread.address, args.length, 1, 0, 0, null);
                if (status === exports.LuaReturn.Yield) {
                    throw new Error('cannot yield in callbacks from javascript');
                }
                thread.assertOk(status);
                const result = thread.getValue(-1);
                thread.pop();
                return result;
            };
            (_a = this.functionRegistry) === null || _a === void 0 ? void 0 : _a.register(jsFunc, func);
            return jsFunc;
        }
    }
    function createTypeExtension$4(thread) {
        return new FunctionTypeExtension(thread);
    }

    class PromiseTypeExtension extends LuaTypeExtension {
        constructor(thread, injectObject) {
            super(thread, 'js_promise');
            this.gcPointer = thread.lua.module.addFunction((functionStateAddress) => {
                const userDataPointer = thread.lua.luaL_checkudata(functionStateAddress, 1, this.name);
                const referencePointer = thread.lua.module.getValue(userDataPointer, '*');
                thread.lua.unref(referencePointer);
                return exports.LuaReturn.Ok;
            }, 'ii');
            if (thread.lua.luaL_newmetatable(thread.address, this.name)) {
                const metatableIndex = thread.lua.lua_gettop(thread.address);
                thread.lua.lua_pushstring(thread.address, 'protected metatable');
                thread.lua.lua_setfield(thread.address, metatableIndex, '__metatable');
                thread.lua.lua_pushcclosure(thread.address, this.gcPointer, 0);
                thread.lua.lua_setfield(thread.address, metatableIndex, '__gc');
                const checkSelf = (self) => {
                    if (Promise.resolve(self) !== self) {
                        throw new Error('promise method called without self instance');
                    }
                    return true;
                };
                thread.pushValue({
                    next: (self, ...args) => checkSelf(self) && self.then(...args),
                    catch: (self, ...args) => checkSelf(self) && self.catch(...args),
                    finally: (self, ...args) => checkSelf(self) && self.finally(...args),
                    await: decorateFunction((functionThread, self) => {
                        checkSelf(self);
                        if (functionThread.address === thread.address) {
                            throw new Error('cannot await in the main thread');
                        }
                        let promiseResult = undefined;
                        const awaitPromise = self
                            .then((res) => {
                            promiseResult = { status: 'fulfilled', value: res };
                        })
                            .catch((err) => {
                            promiseResult = { status: 'rejected', value: err };
                        });
                        const continuance = this.thread.lua.module.addFunction((continuanceState) => {
                            if (!promiseResult) {
                                return thread.lua.lua_yieldk(functionThread.address, 0, 0, continuance);
                            }
                            this.thread.lua.module.removeFunction(continuance);
                            const continuanceThread = thread.stateToThread(continuanceState);
                            if (promiseResult.status === 'rejected') {
                                continuanceThread.pushValue(promiseResult.value || new Error('promise rejected with no error'));
                                return this.thread.lua.lua_error(continuanceState);
                            }
                            if (promiseResult.value instanceof RawResult) {
                                return promiseResult.value.count;
                            }
                            else if (promiseResult.value instanceof MultiReturn) {
                                for (const arg of promiseResult.value) {
                                    continuanceThread.pushValue(arg);
                                }
                                return promiseResult.value.length;
                            }
                            else {
                                continuanceThread.pushValue(promiseResult.value);
                                return 1;
                            }
                        }, 'iiii');
                        functionThread.pushValue(awaitPromise);
                        return new RawResult(thread.lua.lua_yieldk(functionThread.address, 1, 0, continuance));
                    }, { receiveThread: true }),
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__index');
                thread.pushValue((self, other) => self === other);
                thread.lua.lua_setfield(thread.address, metatableIndex, '__eq');
            }
            thread.lua.lua_pop(thread.address, 1);
            if (injectObject) {
                thread.set('Promise', {
                    create: (callback) => new Promise(callback),
                    all: (promiseArray) => {
                        if (!Array.isArray(promiseArray)) {
                            throw new Error('argument must be an array of promises');
                        }
                        return Promise.all(promiseArray.map((potentialPromise) => Promise.resolve(potentialPromise)));
                    },
                    resolve: (value) => Promise.resolve(value),
                });
            }
        }
        close() {
            this.thread.lua.module.removeFunction(this.gcPointer);
        }
        pushValue(thread, decoration) {
            if (Promise.resolve(decoration.target) !== decoration.target) {
                return false;
            }
            return super.pushValue(thread, decoration);
        }
    }
    function createTypeExtension$3(thread, injectObject) {
        return new PromiseTypeExtension(thread, injectObject);
    }

    function decorateProxy(target, options) {
        return new Decoration(target, options || {});
    }
    class ProxyTypeExtension extends LuaTypeExtension {
        constructor(thread) {
            super(thread, 'js_proxy');
            this.gcPointer = thread.lua.module.addFunction((functionStateAddress) => {
                const userDataPointer = thread.lua.luaL_checkudata(functionStateAddress, 1, this.name);
                const referencePointer = thread.lua.module.getValue(userDataPointer, '*');
                thread.lua.unref(referencePointer);
                return exports.LuaReturn.Ok;
            }, 'ii');
            if (thread.lua.luaL_newmetatable(thread.address, this.name)) {
                const metatableIndex = thread.lua.lua_gettop(thread.address);
                thread.lua.lua_pushstring(thread.address, 'protected metatable');
                thread.lua.lua_setfield(thread.address, metatableIndex, '__metatable');
                thread.lua.lua_pushcclosure(thread.address, this.gcPointer, 0);
                thread.lua.lua_setfield(thread.address, metatableIndex, '__gc');
                thread.pushValue((self, key) => {
                    switch (typeof key) {
                        case 'number':
                            key = key - 1;
                        case 'string':
                            break;
                        default:
                            throw new Error('Only strings or numbers can index js objects');
                    }
                    const value = self[key];
                    if (typeof value === 'function') {
                        return decorateFunction(value, { self });
                    }
                    return value;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__index');
                thread.pushValue((self, key, value) => {
                    switch (typeof key) {
                        case 'number':
                            key = key - 1;
                        case 'string':
                            break;
                        default:
                            throw new Error('Only strings or numbers can index js objects');
                    }
                    self[key] = value;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__newindex');
                thread.pushValue((self) => {
                    var _a, _b;
                    return (_b = (_a = self.toString) === null || _a === void 0 ? void 0 : _a.call(self)) !== null && _b !== void 0 ? _b : typeof self;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__tostring');
                thread.pushValue((self) => {
                    return self.length || 0;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__len');
                thread.pushValue((self) => {
                    const keys = Object.getOwnPropertyNames(self);
                    let i = 0;
                    return MultiReturn.of(() => {
                        const ret = MultiReturn.of(keys[i], self[keys[i]]);
                        i++;
                        return ret;
                    }, self, null);
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__pairs');
                thread.pushValue((self, other) => {
                    return self === other;
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__eq');
                thread.pushValue((self, ...args) => {
                    if (args[0] === self) {
                        args.shift();
                    }
                    return self(...args);
                });
                thread.lua.lua_setfield(thread.address, metatableIndex, '__call');
            }
            thread.lua.lua_pop(thread.address, 1);
        }
        isType(_thread, _index, type, name) {
            return type === exports.LuaType.Userdata && name === this.name;
        }
        getValue(thread, index) {
            const refUserdata = thread.lua.lua_touserdata(thread.address, index);
            const referencePointer = thread.lua.module.getValue(refUserdata, '*');
            return thread.lua.getRef(referencePointer);
        }
        pushValue(thread, decoratedValue) {
            var _a;
            const { target, options } = decoratedValue;
            if (options.proxy === undefined) {
                if (target === null || target === undefined) {
                    return false;
                }
                if (typeof target !== 'object') {
                    const isClass = typeof target === 'function' && ((_a = target.prototype) === null || _a === void 0 ? void 0 : _a.constructor) === target && target.toString().startsWith('class ');
                    if (!isClass) {
                        return false;
                    }
                }
                if (Promise.resolve(target) === target) {
                    return false;
                }
            }
            else if (options.proxy === false) {
                return false;
            }
            if (options.metatable && !(options.metatable instanceof Decoration)) {
                decoratedValue.options.metatable = decorateProxy(options.metatable, { proxy: false });
                return false;
            }
            return super.pushValue(thread, decoratedValue);
        }
        close() {
            this.thread.lua.module.removeFunction(this.gcPointer);
        }
    }
    function createTypeExtension$2(thread) {
        return new ProxyTypeExtension(thread);
    }

    class TableTypeExtension extends LuaTypeExtension {
        constructor(thread) {
            super(thread, 'js_table');
        }
        close() {
        }
        isType(_thread, _index, type) {
            return type === exports.LuaType.Table;
        }
        getValue(thread, index, userdata) {
            const seenMap = userdata || new Map();
            const pointer = thread.lua.lua_topointer(thread.address, index);
            let table = seenMap.get(pointer);
            if (!table) {
                const keys = this.readTableKeys(thread, index);
                const isSequential = keys.length > 0 && keys.every((key, index) => key === String(index + 1));
                table = isSequential ? [] : {};
                seenMap.set(pointer, table);
                this.readTableValues(thread, index, seenMap, table);
            }
            return table;
        }
        pushValue(thread, { target }, userdata) {
            if (typeof target !== 'object' || target === null) {
                return false;
            }
            const seenMap = userdata || new Map();
            const existingReference = seenMap.get(target);
            if (existingReference !== undefined) {
                thread.lua.lua_rawgeti(thread.address, LUA_REGISTRYINDEX, existingReference);
                return true;
            }
            try {
                const tableIndex = thread.getTop() + 1;
                const createTable = (arrayCount, keyCount) => {
                    thread.lua.lua_createtable(thread.address, arrayCount, keyCount);
                    const ref = thread.lua.luaL_ref(thread.address, LUA_REGISTRYINDEX);
                    seenMap.set(target, ref);
                    thread.lua.lua_rawgeti(thread.address, LUA_REGISTRYINDEX, ref);
                };
                if (Array.isArray(target)) {
                    createTable(target.length, 0);
                    for (let i = 0; i < target.length; i++) {
                        thread.pushValue(i + 1, seenMap);
                        thread.pushValue(target[i], seenMap);
                        thread.lua.lua_settable(thread.address, tableIndex);
                    }
                }
                else {
                    createTable(0, Object.getOwnPropertyNames(target).length);
                    for (const key in target) {
                        thread.pushValue(key, seenMap);
                        thread.pushValue(target[key], seenMap);
                        thread.lua.lua_settable(thread.address, tableIndex);
                    }
                }
            }
            finally {
                if (userdata === undefined) {
                    for (const reference of seenMap.values()) {
                        thread.lua.luaL_unref(thread.address, LUA_REGISTRYINDEX, reference);
                    }
                }
            }
            return true;
        }
        readTableKeys(thread, index) {
            const keys = [];
            thread.lua.lua_pushnil(thread.address);
            while (thread.lua.lua_next(thread.address, index)) {
                const key = thread.indexToString(-2);
                keys.push(key);
                thread.pop();
            }
            return keys;
        }
        readTableValues(thread, index, seenMap, table) {
            const isArray = Array.isArray(table);
            thread.lua.lua_pushnil(thread.address);
            while (thread.lua.lua_next(thread.address, index)) {
                const key = thread.indexToString(-2);
                const value = thread.getValue(-1, undefined, seenMap);
                if (isArray) {
                    table.push(value);
                }
                else {
                    table[key] = value;
                }
                thread.pop();
            }
        }
    }
    function createTypeExtension$1(thread) {
        return new TableTypeExtension(thread);
    }

    function decorateUserdata(target) {
        return new Decoration(target, { reference: true });
    }
    class UserdataTypeExtension extends LuaTypeExtension {
        constructor(thread) {
            super(thread, 'js_userdata');
            this.gcPointer = thread.lua.module.addFunction((functionStateAddress) => {
                const userDataPointer = thread.lua.luaL_checkudata(functionStateAddress, 1, this.name);
                const referencePointer = thread.lua.module.getValue(userDataPointer, '*');
                thread.lua.unref(referencePointer);
                return exports.LuaReturn.Ok;
            }, 'ii');
            if (thread.lua.luaL_newmetatable(thread.address, this.name)) {
                const metatableIndex = thread.lua.lua_gettop(thread.address);
                thread.lua.lua_pushstring(thread.address, 'protected metatable');
                thread.lua.lua_setfield(thread.address, metatableIndex, '__metatable');
                thread.lua.lua_pushcclosure(thread.address, this.gcPointer, 0);
                thread.lua.lua_setfield(thread.address, metatableIndex, '__gc');
            }
            thread.lua.lua_pop(thread.address, 1);
        }
        isType(_thread, _index, type, name) {
            return type === exports.LuaType.Userdata && name === this.name;
        }
        getValue(thread, index) {
            const refUserdata = thread.lua.lua_touserdata(thread.address, index);
            const referencePointer = thread.lua.module.getValue(refUserdata, '*');
            return thread.lua.getRef(referencePointer);
        }
        pushValue(thread, decoratedValue) {
            if (!decoratedValue.options.reference) {
                return false;
            }
            return super.pushValue(thread, decoratedValue);
        }
        close() {
            this.thread.lua.module.removeFunction(this.gcPointer);
        }
    }
    function createTypeExtension(thread) {
        return new UserdataTypeExtension(thread);
    }

    class LuaEngine {
        constructor(cmodule, { openStandardLibs = true, injectObjects = false, enableProxy = true, traceAllocations = false } = {}) {
            this.cmodule = cmodule;
            this.global = new Global(this.cmodule, traceAllocations);
            this.global.registerTypeExtension(0, createTypeExtension$1(this.global));
            this.global.registerTypeExtension(0, createTypeExtension$4(this.global));
            this.global.registerTypeExtension(1, createTypeExtension$3(this.global, injectObjects));
            if (enableProxy) {
                this.global.registerTypeExtension(3, createTypeExtension$2(this.global));
            }
            else {
                this.global.registerTypeExtension(1, createTypeExtension$5(this.global, injectObjects));
            }
            this.global.registerTypeExtension(4, createTypeExtension(this.global));
            if (openStandardLibs) {
                this.cmodule.luaL_openlibs(this.global.address);
            }
        }
        doString(script) {
            return this.callByteCode((thread) => thread.loadString(script));
        }
        doFile(filename) {
            return this.callByteCode((thread) => thread.loadFile(filename));
        }
        doStringSync(script) {
            this.global.loadString(script);
            const result = this.global.runSync();
            return result[0];
        }
        doFileSync(filename) {
            this.global.loadFile(filename);
            const result = this.global.runSync();
            return result[0];
        }
        doBytecodeSync(bytecode, args) {
            this.global.loadBytecode(bytecode);
            if (typeof args === 'undefined') {
                return this.global.runSync();
            }
            for (const arg of args) {
                this.global.pushValue(arg);
            }
            return this.global.runSync(args.length);
        }
        async callByteCode(loader) {
            const thread = this.global.newThread();
            const threadIndex = this.global.getTop();
            try {
                loader(thread);
                const result = await thread.run(0);
                if (result.length > 0) {
                    this.cmodule.lua_xmove(thread.address, this.global.address, result.length);
                }
                return result[0];
            }
            finally {
                this.global.remove(threadIndex);
            }
        }
    }

    var initWasmModule = (() => {
      var _scriptDir = (typeof document === 'undefined' && typeof location === 'undefined' ? new (require('u' + 'rl').URL)('file:' + __filename).href : typeof document === 'undefined' ? location.href : (document.currentScript && document.currentScript.src || new URL('index.js', document.baseURI).href));
      
      return (
    async function(moduleArg = {}) {

    var Module=moduleArg;var readyPromiseResolve,readyPromiseReject;Module["ready"]=new Promise(((resolve,reject)=>{readyPromiseResolve=resolve;readyPromiseReject=reject;}));var moduleOverrides=Object.assign({},Module);var thisProgram="./this.program";var quit_=(status,toThrow)=>{throw toThrow};var ENVIRONMENT_IS_WEB=typeof window=="object";var ENVIRONMENT_IS_WORKER=typeof importScripts=="function";var ENVIRONMENT_IS_NODE=typeof process=="object"&&typeof process.versions=="object"&&typeof process.versions.node=="string";var scriptDirectory="";function locateFile(path){if(Module["locateFile"]){return Module["locateFile"](path,scriptDirectory)}return scriptDirectory+path}var read_,readAsync,readBinary;if(ENVIRONMENT_IS_NODE){const{createRequire:createRequire}=await import('module');var require$1=createRequire((typeof document === 'undefined' && typeof location === 'undefined' ? new (require('u' + 'rl').URL)('file:' + __filename).href : typeof document === 'undefined' ? location.href : (document.currentScript && document.currentScript.src || new URL('index.js', document.baseURI).href)));var fs=require$1("fs");var nodePath=require$1("path");if(ENVIRONMENT_IS_WORKER){scriptDirectory=nodePath.dirname(scriptDirectory)+"/";}else {scriptDirectory=require$1("url").fileURLToPath(new URL("./",(typeof document === 'undefined' && typeof location === 'undefined' ? new (require('u' + 'rl').URL)('file:' + __filename).href : typeof document === 'undefined' ? location.href : (document.currentScript && document.currentScript.src || new URL('index.js', document.baseURI).href))));}read_=(filename,binary)=>{filename=isFileURI(filename)?new URL(filename):nodePath.normalize(filename);return fs.readFileSync(filename,binary?undefined:"utf8")};readBinary=filename=>{var ret=read_(filename,true);if(!ret.buffer){ret=new Uint8Array(ret);}return ret};readAsync=(filename,onload,onerror,binary=true)=>{filename=isFileURI(filename)?new URL(filename):nodePath.normalize(filename);fs.readFile(filename,binary?undefined:"utf8",((err,data)=>{if(err)onerror(err);else onload(binary?data.buffer:data);}));};if(!Module["thisProgram"]&&process.argv.length>1){thisProgram=process.argv[1].replace(/\\/g,"/");}process.argv.slice(2);quit_=(status,toThrow)=>{process.exitCode=status;throw toThrow};Module["inspect"]=()=>"[Emscripten Module object]";}else if(ENVIRONMENT_IS_WEB||ENVIRONMENT_IS_WORKER){if(ENVIRONMENT_IS_WORKER){scriptDirectory=self.location.href;}else if(typeof document!="undefined"&&document.currentScript){scriptDirectory=document.currentScript.src;}if(_scriptDir){scriptDirectory=_scriptDir;}if(scriptDirectory.indexOf("blob:")!==0){scriptDirectory=scriptDirectory.substr(0,scriptDirectory.replace(/[?#].*/,"").lastIndexOf("/")+1);}else {scriptDirectory="";}{read_=url=>{var xhr=new XMLHttpRequest;xhr.open("GET",url,false);xhr.send(null);return xhr.responseText};if(ENVIRONMENT_IS_WORKER){readBinary=url=>{var xhr=new XMLHttpRequest;xhr.open("GET",url,false);xhr.responseType="arraybuffer";xhr.send(null);return new Uint8Array(xhr.response)};}readAsync=(url,onload,onerror)=>{var xhr=new XMLHttpRequest;xhr.open("GET",url,true);xhr.responseType="arraybuffer";xhr.onload=()=>{if(xhr.status==200||xhr.status==0&&xhr.response){onload(xhr.response);return}onerror();};xhr.onerror=onerror;xhr.send(null);};}}else;var out=Module["print"]||console.log.bind(console);var err=Module["printErr"]||console.error.bind(console);Object.assign(Module,moduleOverrides);moduleOverrides=null;if(Module["arguments"])Module["arguments"];if(Module["thisProgram"])thisProgram=Module["thisProgram"];if(Module["quit"])quit_=Module["quit"];var wasmBinary;if(Module["wasmBinary"])wasmBinary=Module["wasmBinary"];var noExitRuntime=Module["noExitRuntime"]||true;if(typeof WebAssembly!="object"){abort("no native wasm support detected");}var wasmMemory;var ABORT=false;function assert(condition,text){if(!condition){abort(text);}}var HEAP8,HEAPU8,HEAP16,HEAP32,HEAPU32,HEAPF32,HEAPF64;function updateMemoryViews(){var b=wasmMemory.buffer;Module["HEAP8"]=HEAP8=new Int8Array(b);Module["HEAP16"]=HEAP16=new Int16Array(b);Module["HEAP32"]=HEAP32=new Int32Array(b);Module["HEAPU8"]=HEAPU8=new Uint8Array(b);Module["HEAPU16"]=new Uint16Array(b);Module["HEAPU32"]=HEAPU32=new Uint32Array(b);Module["HEAPF32"]=HEAPF32=new Float32Array(b);Module["HEAPF64"]=HEAPF64=new Float64Array(b);}var wasmTable;var __ATPRERUN__=[];var __ATINIT__=[];var __ATPOSTRUN__=[];var runtimeKeepaliveCounter=0;function keepRuntimeAlive(){return noExitRuntime||runtimeKeepaliveCounter>0}function preRun(){if(Module["preRun"]){if(typeof Module["preRun"]=="function")Module["preRun"]=[Module["preRun"]];while(Module["preRun"].length){addOnPreRun(Module["preRun"].shift());}}callRuntimeCallbacks(__ATPRERUN__);}function initRuntime(){if(!Module["noFSInit"]&&!FS.init.initialized)FS.init();FS.ignorePermissions=false;callRuntimeCallbacks(__ATINIT__);}function postRun(){if(Module["postRun"]){if(typeof Module["postRun"]=="function")Module["postRun"]=[Module["postRun"]];while(Module["postRun"].length){addOnPostRun(Module["postRun"].shift());}}callRuntimeCallbacks(__ATPOSTRUN__);}function addOnPreRun(cb){__ATPRERUN__.unshift(cb);}function addOnInit(cb){__ATINIT__.unshift(cb);}function addOnPostRun(cb){__ATPOSTRUN__.unshift(cb);}var runDependencies=0;var dependenciesFulfilled=null;function getUniqueRunDependency(id){return id}function addRunDependency(id){runDependencies++;if(Module["monitorRunDependencies"]){Module["monitorRunDependencies"](runDependencies);}}function removeRunDependency(id){runDependencies--;if(Module["monitorRunDependencies"]){Module["monitorRunDependencies"](runDependencies);}if(runDependencies==0){if(dependenciesFulfilled){var callback=dependenciesFulfilled;dependenciesFulfilled=null;callback();}}}function abort(what){if(Module["onAbort"]){Module["onAbort"](what);}what="Aborted("+what+")";err(what);ABORT=true;what+=". Build with -sASSERTIONS for more info.";var e=new WebAssembly.RuntimeError(what);readyPromiseReject(e);throw e}var dataURIPrefix="data:application/octet-stream;base64,";function isDataURI(filename){return filename.startsWith(dataURIPrefix)}function isFileURI(filename){return filename.startsWith("file://")}var wasmBinaryFile;if(Module["locateFile"]){wasmBinaryFile="glue.wasm";if(!isDataURI(wasmBinaryFile)){wasmBinaryFile=locateFile(wasmBinaryFile);}}else {wasmBinaryFile=new URL("glue.wasm",(typeof document === 'undefined' && typeof location === 'undefined' ? new (require('u' + 'rl').URL)('file:' + __filename).href : typeof document === 'undefined' ? location.href : (document.currentScript && document.currentScript.src || new URL('index.js', document.baseURI).href))).href;}function getBinarySync(file){if(file==wasmBinaryFile&&wasmBinary){return new Uint8Array(wasmBinary)}if(readBinary){return readBinary(file)}throw "both async and sync fetching of the wasm failed"}function getBinaryPromise(binaryFile){if(!wasmBinary&&(ENVIRONMENT_IS_WEB||ENVIRONMENT_IS_WORKER)){if(typeof fetch=="function"&&!isFileURI(binaryFile)){return fetch(binaryFile,{credentials:"same-origin"}).then((response=>{if(!response["ok"]){throw "failed to load wasm binary file at '"+binaryFile+"'"}return response["arrayBuffer"]()})).catch((()=>getBinarySync(binaryFile)))}else if(readAsync){return new Promise(((resolve,reject)=>{readAsync(binaryFile,(response=>resolve(new Uint8Array(response))),reject);}))}}return Promise.resolve().then((()=>getBinarySync(binaryFile)))}function instantiateArrayBuffer(binaryFile,imports,receiver){return getBinaryPromise(binaryFile).then((binary=>WebAssembly.instantiate(binary,imports))).then((instance=>instance)).then(receiver,(reason=>{err("failed to asynchronously prepare wasm: "+reason);abort(reason);}))}function instantiateAsync(binary,binaryFile,imports,callback){if(!binary&&typeof WebAssembly.instantiateStreaming=="function"&&!isDataURI(binaryFile)&&!isFileURI(binaryFile)&&!ENVIRONMENT_IS_NODE&&typeof fetch=="function"){return fetch(binaryFile,{credentials:"same-origin"}).then((response=>{var result=WebAssembly.instantiateStreaming(response,imports);return result.then(callback,(function(reason){err("wasm streaming compile failed: "+reason);err("falling back to ArrayBuffer instantiation");return instantiateArrayBuffer(binaryFile,imports,callback)}))}))}return instantiateArrayBuffer(binaryFile,imports,callback)}function createWasm(){var info={"a":wasmImports};function receiveInstance(instance,module){var exports=instance.exports;Module["asm"]=exports;wasmMemory=Module["asm"]["C"];updateMemoryViews();wasmTable=Module["asm"]["ab"];addOnInit(Module["asm"]["D"]);removeRunDependency();return exports}addRunDependency();function receiveInstantiationResult(result){receiveInstance(result["instance"]);}if(Module["instantiateWasm"]){try{return Module["instantiateWasm"](info,receiveInstance)}catch(e){err("Module.instantiateWasm callback failed with error: "+e);readyPromiseReject(e);}}instantiateAsync(wasmBinary,wasmBinaryFile,info,receiveInstantiationResult).catch(readyPromiseReject);return {}}var tempDouble;var tempI64;function ExitStatus(status){this.name="ExitStatus";this.message=`Program terminated with exit(${status})`;this.status=status;}var callRuntimeCallbacks=callbacks=>{while(callbacks.length>0){callbacks.shift()(Module);}};function getValue(ptr,type="i8"){if(type.endsWith("*"))type="*";switch(type){case"i1":return HEAP8[ptr>>0];case"i8":return HEAP8[ptr>>0];case"i16":return HEAP16[ptr>>1];case"i32":return HEAP32[ptr>>2];case"i64":abort("to do getValue(i64) use WASM_BIGINT");case"float":return HEAPF32[ptr>>2];case"double":return HEAPF64[ptr>>3];case"*":return HEAPU32[ptr>>2];default:abort(`invalid type for getValue: ${type}`);}}function setValue(ptr,value,type="i8"){if(type.endsWith("*"))type="*";switch(type){case"i1":HEAP8[ptr>>0]=value;break;case"i8":HEAP8[ptr>>0]=value;break;case"i16":HEAP16[ptr>>1]=value;break;case"i32":HEAP32[ptr>>2]=value;break;case"i64":abort("to do setValue(i64) use WASM_BIGINT");case"float":HEAPF32[ptr>>2]=value;break;case"double":HEAPF64[ptr>>3]=value;break;case"*":HEAPU32[ptr>>2]=value;break;default:abort(`invalid type for setValue: ${type}`);}}var PATH={isAbs:path=>path.charAt(0)==="/",splitPath:filename=>{var splitPathRe=/^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/;return splitPathRe.exec(filename).slice(1)},normalizeArray:(parts,allowAboveRoot)=>{var up=0;for(var i=parts.length-1;i>=0;i--){var last=parts[i];if(last==="."){parts.splice(i,1);}else if(last===".."){parts.splice(i,1);up++;}else if(up){parts.splice(i,1);up--;}}if(allowAboveRoot){for(;up;up--){parts.unshift("..");}}return parts},normalize:path=>{var isAbsolute=PATH.isAbs(path),trailingSlash=path.substr(-1)==="/";path=PATH.normalizeArray(path.split("/").filter((p=>!!p)),!isAbsolute).join("/");if(!path&&!isAbsolute){path=".";}if(path&&trailingSlash){path+="/";}return (isAbsolute?"/":"")+path},dirname:path=>{var result=PATH.splitPath(path),root=result[0],dir=result[1];if(!root&&!dir){return "."}if(dir){dir=dir.substr(0,dir.length-1);}return root+dir},basename:path=>{if(path==="/")return "/";path=PATH.normalize(path);path=path.replace(/\/$/,"");var lastSlash=path.lastIndexOf("/");if(lastSlash===-1)return path;return path.substr(lastSlash+1)},join:function(){var paths=Array.prototype.slice.call(arguments);return PATH.normalize(paths.join("/"))},join2:(l,r)=>PATH.normalize(l+"/"+r)};var initRandomFill=()=>{if(typeof crypto=="object"&&typeof crypto["getRandomValues"]=="function"){return view=>crypto.getRandomValues(view)}else if(ENVIRONMENT_IS_NODE){try{var crypto_module=require$1("crypto");var randomFillSync=crypto_module["randomFillSync"];if(randomFillSync){return view=>crypto_module["randomFillSync"](view)}var randomBytes=crypto_module["randomBytes"];return view=>(view.set(randomBytes(view.byteLength)),view)}catch(e){}}abort("initRandomDevice");};var randomFill=view=>(randomFill=initRandomFill())(view);var PATH_FS={resolve:function(){var resolvedPath="",resolvedAbsolute=false;for(var i=arguments.length-1;i>=-1&&!resolvedAbsolute;i--){var path=i>=0?arguments[i]:FS.cwd();if(typeof path!="string"){throw new TypeError("Arguments to path.resolve must be strings")}else if(!path){return ""}resolvedPath=path+"/"+resolvedPath;resolvedAbsolute=PATH.isAbs(path);}resolvedPath=PATH.normalizeArray(resolvedPath.split("/").filter((p=>!!p)),!resolvedAbsolute).join("/");return (resolvedAbsolute?"/":"")+resolvedPath||"."},relative:(from,to)=>{from=PATH_FS.resolve(from).substr(1);to=PATH_FS.resolve(to).substr(1);function trim(arr){var start=0;for(;start<arr.length;start++){if(arr[start]!=="")break}var end=arr.length-1;for(;end>=0;end--){if(arr[end]!=="")break}if(start>end)return [];return arr.slice(start,end-start+1)}var fromParts=trim(from.split("/"));var toParts=trim(to.split("/"));var length=Math.min(fromParts.length,toParts.length);var samePartsLength=length;for(var i=0;i<length;i++){if(fromParts[i]!==toParts[i]){samePartsLength=i;break}}var outputParts=[];for(var i=samePartsLength;i<fromParts.length;i++){outputParts.push("..");}outputParts=outputParts.concat(toParts.slice(samePartsLength));return outputParts.join("/")}};var UTF8Decoder=typeof TextDecoder!="undefined"?new TextDecoder("utf8"):undefined;var UTF8ArrayToString=(heapOrArray,idx,maxBytesToRead)=>{var endIdx=idx+maxBytesToRead;var endPtr=idx;while(heapOrArray[endPtr]&&!(endPtr>=endIdx))++endPtr;if(endPtr-idx>16&&heapOrArray.buffer&&UTF8Decoder){return UTF8Decoder.decode(heapOrArray.subarray(idx,endPtr))}var str="";while(idx<endPtr){var u0=heapOrArray[idx++];if(!(u0&128)){str+=String.fromCharCode(u0);continue}var u1=heapOrArray[idx++]&63;if((u0&224)==192){str+=String.fromCharCode((u0&31)<<6|u1);continue}var u2=heapOrArray[idx++]&63;if((u0&240)==224){u0=(u0&15)<<12|u1<<6|u2;}else {u0=(u0&7)<<18|u1<<12|u2<<6|heapOrArray[idx++]&63;}if(u0<65536){str+=String.fromCharCode(u0);}else {var ch=u0-65536;str+=String.fromCharCode(55296|ch>>10,56320|ch&1023);}}return str};var FS_stdin_getChar_buffer=[];var lengthBytesUTF8=str=>{var len=0;for(var i=0;i<str.length;++i){var c=str.charCodeAt(i);if(c<=127){len++;}else if(c<=2047){len+=2;}else if(c>=55296&&c<=57343){len+=4;++i;}else {len+=3;}}return len};var stringToUTF8Array=(str,heap,outIdx,maxBytesToWrite)=>{if(!(maxBytesToWrite>0))return 0;var startIdx=outIdx;var endIdx=outIdx+maxBytesToWrite-1;for(var i=0;i<str.length;++i){var u=str.charCodeAt(i);if(u>=55296&&u<=57343){var u1=str.charCodeAt(++i);u=65536+((u&1023)<<10)|u1&1023;}if(u<=127){if(outIdx>=endIdx)break;heap[outIdx++]=u;}else if(u<=2047){if(outIdx+1>=endIdx)break;heap[outIdx++]=192|u>>6;heap[outIdx++]=128|u&63;}else if(u<=65535){if(outIdx+2>=endIdx)break;heap[outIdx++]=224|u>>12;heap[outIdx++]=128|u>>6&63;heap[outIdx++]=128|u&63;}else {if(outIdx+3>=endIdx)break;heap[outIdx++]=240|u>>18;heap[outIdx++]=128|u>>12&63;heap[outIdx++]=128|u>>6&63;heap[outIdx++]=128|u&63;}}heap[outIdx]=0;return outIdx-startIdx};function intArrayFromString(stringy,dontAddNull,length){var len=length>0?length:lengthBytesUTF8(stringy)+1;var u8array=new Array(len);var numBytesWritten=stringToUTF8Array(stringy,u8array,0,u8array.length);if(dontAddNull)u8array.length=numBytesWritten;return u8array}var FS_stdin_getChar=()=>{if(!FS_stdin_getChar_buffer.length){var result=null;if(ENVIRONMENT_IS_NODE){var BUFSIZE=256;var buf=Buffer.alloc(BUFSIZE);var bytesRead=0;var fd=process.stdin.fd;try{bytesRead=fs.readSync(fd,buf,0,BUFSIZE,-1);}catch(e){if(e.toString().includes("EOF"))bytesRead=0;else throw e}if(bytesRead>0){result=buf.slice(0,bytesRead).toString("utf-8");}else {result=null;}}else if(typeof window!="undefined"&&typeof window.prompt=="function"){result=window.prompt("Input: ");if(result!==null){result+="\n";}}else if(typeof readline=="function"){result=readline();if(result!==null){result+="\n";}}if(!result){return null}FS_stdin_getChar_buffer=intArrayFromString(result,true);}return FS_stdin_getChar_buffer.shift()};var TTY={ttys:[],init:function(){},shutdown:function(){},register:function(dev,ops){TTY.ttys[dev]={input:[],output:[],ops:ops};FS.registerDevice(dev,TTY.stream_ops);},stream_ops:{open:function(stream){var tty=TTY.ttys[stream.node.rdev];if(!tty){throw new FS.ErrnoError(43)}stream.tty=tty;stream.seekable=false;},close:function(stream){stream.tty.ops.fsync(stream.tty);},fsync:function(stream){stream.tty.ops.fsync(stream.tty);},read:function(stream,buffer,offset,length,pos){if(!stream.tty||!stream.tty.ops.get_char){throw new FS.ErrnoError(60)}var bytesRead=0;for(var i=0;i<length;i++){var result;try{result=stream.tty.ops.get_char(stream.tty);}catch(e){throw new FS.ErrnoError(29)}if(result===undefined&&bytesRead===0){throw new FS.ErrnoError(6)}if(result===null||result===undefined)break;bytesRead++;buffer[offset+i]=result;}if(bytesRead){stream.node.timestamp=Date.now();}return bytesRead},write:function(stream,buffer,offset,length,pos){if(!stream.tty||!stream.tty.ops.put_char){throw new FS.ErrnoError(60)}try{for(var i=0;i<length;i++){stream.tty.ops.put_char(stream.tty,buffer[offset+i]);}}catch(e){throw new FS.ErrnoError(29)}if(length){stream.node.timestamp=Date.now();}return i}},default_tty_ops:{get_char:function(tty){return FS_stdin_getChar()},put_char:function(tty,val){if(val===null||val===10){out(UTF8ArrayToString(tty.output,0));tty.output=[];}else {if(val!=0)tty.output.push(val);}},fsync:function(tty){if(tty.output&&tty.output.length>0){out(UTF8ArrayToString(tty.output,0));tty.output=[];}},ioctl_tcgets:function(tty){return {c_iflag:25856,c_oflag:5,c_cflag:191,c_lflag:35387,c_cc:[3,28,127,21,4,0,1,0,17,19,26,0,18,15,23,22,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]}},ioctl_tcsets:function(tty,optional_actions,data){return 0},ioctl_tiocgwinsz:function(tty){return [24,80]}},default_tty1_ops:{put_char:function(tty,val){if(val===null||val===10){err(UTF8ArrayToString(tty.output,0));tty.output=[];}else {if(val!=0)tty.output.push(val);}},fsync:function(tty){if(tty.output&&tty.output.length>0){err(UTF8ArrayToString(tty.output,0));tty.output=[];}}}};var mmapAlloc=size=>{abort();};var MEMFS={ops_table:null,mount(mount){return MEMFS.createNode(null,"/",16384|511,0)},createNode(parent,name,mode,dev){if(FS.isBlkdev(mode)||FS.isFIFO(mode)){throw new FS.ErrnoError(63)}if(!MEMFS.ops_table){MEMFS.ops_table={dir:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr,lookup:MEMFS.node_ops.lookup,mknod:MEMFS.node_ops.mknod,rename:MEMFS.node_ops.rename,unlink:MEMFS.node_ops.unlink,rmdir:MEMFS.node_ops.rmdir,readdir:MEMFS.node_ops.readdir,symlink:MEMFS.node_ops.symlink},stream:{llseek:MEMFS.stream_ops.llseek}},file:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr},stream:{llseek:MEMFS.stream_ops.llseek,read:MEMFS.stream_ops.read,write:MEMFS.stream_ops.write,allocate:MEMFS.stream_ops.allocate,mmap:MEMFS.stream_ops.mmap,msync:MEMFS.stream_ops.msync}},link:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr,readlink:MEMFS.node_ops.readlink},stream:{}},chrdev:{node:{getattr:MEMFS.node_ops.getattr,setattr:MEMFS.node_ops.setattr},stream:FS.chrdev_stream_ops}};}var node=FS.createNode(parent,name,mode,dev);if(FS.isDir(node.mode)){node.node_ops=MEMFS.ops_table.dir.node;node.stream_ops=MEMFS.ops_table.dir.stream;node.contents={};}else if(FS.isFile(node.mode)){node.node_ops=MEMFS.ops_table.file.node;node.stream_ops=MEMFS.ops_table.file.stream;node.usedBytes=0;node.contents=null;}else if(FS.isLink(node.mode)){node.node_ops=MEMFS.ops_table.link.node;node.stream_ops=MEMFS.ops_table.link.stream;}else if(FS.isChrdev(node.mode)){node.node_ops=MEMFS.ops_table.chrdev.node;node.stream_ops=MEMFS.ops_table.chrdev.stream;}node.timestamp=Date.now();if(parent){parent.contents[name]=node;parent.timestamp=node.timestamp;}return node},getFileDataAsTypedArray(node){if(!node.contents)return new Uint8Array(0);if(node.contents.subarray)return node.contents.subarray(0,node.usedBytes);return new Uint8Array(node.contents)},expandFileStorage(node,newCapacity){var prevCapacity=node.contents?node.contents.length:0;if(prevCapacity>=newCapacity)return;var CAPACITY_DOUBLING_MAX=1024*1024;newCapacity=Math.max(newCapacity,prevCapacity*(prevCapacity<CAPACITY_DOUBLING_MAX?2:1.125)>>>0);if(prevCapacity!=0)newCapacity=Math.max(newCapacity,256);var oldContents=node.contents;node.contents=new Uint8Array(newCapacity);if(node.usedBytes>0)node.contents.set(oldContents.subarray(0,node.usedBytes),0);},resizeFileStorage(node,newSize){if(node.usedBytes==newSize)return;if(newSize==0){node.contents=null;node.usedBytes=0;}else {var oldContents=node.contents;node.contents=new Uint8Array(newSize);if(oldContents){node.contents.set(oldContents.subarray(0,Math.min(newSize,node.usedBytes)));}node.usedBytes=newSize;}},node_ops:{getattr(node){var attr={};attr.dev=FS.isChrdev(node.mode)?node.id:1;attr.ino=node.id;attr.mode=node.mode;attr.nlink=1;attr.uid=0;attr.gid=0;attr.rdev=node.rdev;if(FS.isDir(node.mode)){attr.size=4096;}else if(FS.isFile(node.mode)){attr.size=node.usedBytes;}else if(FS.isLink(node.mode)){attr.size=node.link.length;}else {attr.size=0;}attr.atime=new Date(node.timestamp);attr.mtime=new Date(node.timestamp);attr.ctime=new Date(node.timestamp);attr.blksize=4096;attr.blocks=Math.ceil(attr.size/attr.blksize);return attr},setattr(node,attr){if(attr.mode!==undefined){node.mode=attr.mode;}if(attr.timestamp!==undefined){node.timestamp=attr.timestamp;}if(attr.size!==undefined){MEMFS.resizeFileStorage(node,attr.size);}},lookup(parent,name){throw FS.genericErrors[44]},mknod(parent,name,mode,dev){return MEMFS.createNode(parent,name,mode,dev)},rename(old_node,new_dir,new_name){if(FS.isDir(old_node.mode)){var new_node;try{new_node=FS.lookupNode(new_dir,new_name);}catch(e){}if(new_node){for(var i in new_node.contents){throw new FS.ErrnoError(55)}}}delete old_node.parent.contents[old_node.name];old_node.parent.timestamp=Date.now();old_node.name=new_name;new_dir.contents[new_name]=old_node;new_dir.timestamp=old_node.parent.timestamp;old_node.parent=new_dir;},unlink(parent,name){delete parent.contents[name];parent.timestamp=Date.now();},rmdir(parent,name){var node=FS.lookupNode(parent,name);for(var i in node.contents){throw new FS.ErrnoError(55)}delete parent.contents[name];parent.timestamp=Date.now();},readdir(node){var entries=[".",".."];for(var key in node.contents){if(!node.contents.hasOwnProperty(key)){continue}entries.push(key);}return entries},symlink(parent,newname,oldpath){var node=MEMFS.createNode(parent,newname,511|40960,0);node.link=oldpath;return node},readlink(node){if(!FS.isLink(node.mode)){throw new FS.ErrnoError(28)}return node.link}},stream_ops:{read(stream,buffer,offset,length,position){var contents=stream.node.contents;if(position>=stream.node.usedBytes)return 0;var size=Math.min(stream.node.usedBytes-position,length);if(size>8&&contents.subarray){buffer.set(contents.subarray(position,position+size),offset);}else {for(var i=0;i<size;i++)buffer[offset+i]=contents[position+i];}return size},write(stream,buffer,offset,length,position,canOwn){if(buffer.buffer===HEAP8.buffer){canOwn=false;}if(!length)return 0;var node=stream.node;node.timestamp=Date.now();if(buffer.subarray&&(!node.contents||node.contents.subarray)){if(canOwn){node.contents=buffer.subarray(offset,offset+length);node.usedBytes=length;return length}else if(node.usedBytes===0&&position===0){node.contents=buffer.slice(offset,offset+length);node.usedBytes=length;return length}else if(position+length<=node.usedBytes){node.contents.set(buffer.subarray(offset,offset+length),position);return length}}MEMFS.expandFileStorage(node,position+length);if(node.contents.subarray&&buffer.subarray){node.contents.set(buffer.subarray(offset,offset+length),position);}else {for(var i=0;i<length;i++){node.contents[position+i]=buffer[offset+i];}}node.usedBytes=Math.max(node.usedBytes,position+length);return length},llseek(stream,offset,whence){var position=offset;if(whence===1){position+=stream.position;}else if(whence===2){if(FS.isFile(stream.node.mode)){position+=stream.node.usedBytes;}}if(position<0){throw new FS.ErrnoError(28)}return position},allocate(stream,offset,length){MEMFS.expandFileStorage(stream.node,offset+length);stream.node.usedBytes=Math.max(stream.node.usedBytes,offset+length);},mmap(stream,length,position,prot,flags){if(!FS.isFile(stream.node.mode)){throw new FS.ErrnoError(43)}var ptr;var allocated;var contents=stream.node.contents;if(!(flags&2)&&contents.buffer===HEAP8.buffer){allocated=false;ptr=contents.byteOffset;}else {if(position>0||position+length<contents.length){if(contents.subarray){contents=contents.subarray(position,position+length);}else {contents=Array.prototype.slice.call(contents,position,position+length);}}allocated=true;ptr=mmapAlloc();if(!ptr){throw new FS.ErrnoError(48)}HEAP8.set(contents,ptr);}return {ptr:ptr,allocated:allocated}},msync(stream,buffer,offset,length,mmapFlags){MEMFS.stream_ops.write(stream,buffer,0,length,offset,false);return 0}}};var asyncLoad=(url,onload,onerror,noRunDep)=>{var dep=!noRunDep?getUniqueRunDependency(`al ${url}`):"";readAsync(url,(arrayBuffer=>{assert(arrayBuffer,`Loading data file "${url}" failed (no arrayBuffer).`);onload(new Uint8Array(arrayBuffer));if(dep)removeRunDependency();}),(event=>{if(onerror){onerror();}else {throw `Loading data file "${url}" failed.`}}));if(dep)addRunDependency();};var preloadPlugins=Module["preloadPlugins"]||[];function FS_handledByPreloadPlugin(byteArray,fullname,finish,onerror){if(typeof Browser!="undefined")Browser.init();var handled=false;preloadPlugins.forEach((function(plugin){if(handled)return;if(plugin["canHandle"](fullname)){plugin["handle"](byteArray,fullname,finish,onerror);handled=true;}}));return handled}function FS_createPreloadedFile(parent,name,url,canRead,canWrite,onload,onerror,dontCreateFile,canOwn,preFinish){var fullname=name?PATH_FS.resolve(PATH.join2(parent,name)):parent;function processData(byteArray){function finish(byteArray){if(preFinish)preFinish();if(!dontCreateFile){FS.createDataFile(parent,name,byteArray,canRead,canWrite,canOwn);}if(onload)onload();removeRunDependency();}if(FS_handledByPreloadPlugin(byteArray,fullname,finish,(()=>{if(onerror)onerror();removeRunDependency();}))){return}finish(byteArray);}addRunDependency();if(typeof url=="string"){asyncLoad(url,(byteArray=>processData(byteArray)),onerror);}else {processData(url);}}function FS_modeStringToFlags(str){var flagModes={"r":0,"r+":2,"w":512|64|1,"w+":512|64|2,"a":1024|64|1,"a+":1024|64|2};var flags=flagModes[str];if(typeof flags=="undefined"){throw new Error(`Unknown file open mode: ${str}`)}return flags}function FS_getMode(canRead,canWrite){var mode=0;if(canRead)mode|=292|73;if(canWrite)mode|=146;return mode}var FS={root:null,mounts:[],devices:{},streams:[],nextInode:1,nameTable:null,currentPath:"/",initialized:false,ignorePermissions:true,ErrnoError:null,genericErrors:{},filesystems:null,syncFSRequests:0,lookupPath:(path,opts={})=>{path=PATH_FS.resolve(path);if(!path)return {path:"",node:null};var defaults={follow_mount:true,recurse_count:0};opts=Object.assign(defaults,opts);if(opts.recurse_count>8){throw new FS.ErrnoError(32)}var parts=path.split("/").filter((p=>!!p));var current=FS.root;var current_path="/";for(var i=0;i<parts.length;i++){var islast=i===parts.length-1;if(islast&&opts.parent){break}current=FS.lookupNode(current,parts[i]);current_path=PATH.join2(current_path,parts[i]);if(FS.isMountpoint(current)){if(!islast||islast&&opts.follow_mount){current=current.mounted.root;}}if(!islast||opts.follow){var count=0;while(FS.isLink(current.mode)){var link=FS.readlink(current_path);current_path=PATH_FS.resolve(PATH.dirname(current_path),link);var lookup=FS.lookupPath(current_path,{recurse_count:opts.recurse_count+1});current=lookup.node;if(count++>40){throw new FS.ErrnoError(32)}}}}return {path:current_path,node:current}},getPath:node=>{var path;while(true){if(FS.isRoot(node)){var mount=node.mount.mountpoint;if(!path)return mount;return mount[mount.length-1]!=="/"?`${mount}/${path}`:mount+path}path=path?`${node.name}/${path}`:node.name;node=node.parent;}},hashName:(parentid,name)=>{var hash=0;for(var i=0;i<name.length;i++){hash=(hash<<5)-hash+name.charCodeAt(i)|0;}return (parentid+hash>>>0)%FS.nameTable.length},hashAddNode:node=>{var hash=FS.hashName(node.parent.id,node.name);node.name_next=FS.nameTable[hash];FS.nameTable[hash]=node;},hashRemoveNode:node=>{var hash=FS.hashName(node.parent.id,node.name);if(FS.nameTable[hash]===node){FS.nameTable[hash]=node.name_next;}else {var current=FS.nameTable[hash];while(current){if(current.name_next===node){current.name_next=node.name_next;break}current=current.name_next;}}},lookupNode:(parent,name)=>{var errCode=FS.mayLookup(parent);if(errCode){throw new FS.ErrnoError(errCode,parent)}var hash=FS.hashName(parent.id,name);for(var node=FS.nameTable[hash];node;node=node.name_next){var nodeName=node.name;if(node.parent.id===parent.id&&nodeName===name){return node}}return FS.lookup(parent,name)},createNode:(parent,name,mode,rdev)=>{var node=new FS.FSNode(parent,name,mode,rdev);FS.hashAddNode(node);return node},destroyNode:node=>{FS.hashRemoveNode(node);},isRoot:node=>node===node.parent,isMountpoint:node=>!!node.mounted,isFile:mode=>(mode&61440)===32768,isDir:mode=>(mode&61440)===16384,isLink:mode=>(mode&61440)===40960,isChrdev:mode=>(mode&61440)===8192,isBlkdev:mode=>(mode&61440)===24576,isFIFO:mode=>(mode&61440)===4096,isSocket:mode=>(mode&49152)===49152,flagsToPermissionString:flag=>{var perms=["r","w","rw"][flag&3];if(flag&512){perms+="w";}return perms},nodePermissions:(node,perms)=>{if(FS.ignorePermissions){return 0}if(perms.includes("r")&&!(node.mode&292)){return 2}else if(perms.includes("w")&&!(node.mode&146)){return 2}else if(perms.includes("x")&&!(node.mode&73)){return 2}return 0},mayLookup:dir=>{var errCode=FS.nodePermissions(dir,"x");if(errCode)return errCode;if(!dir.node_ops.lookup)return 2;return 0},mayCreate:(dir,name)=>{try{var node=FS.lookupNode(dir,name);return 20}catch(e){}return FS.nodePermissions(dir,"wx")},mayDelete:(dir,name,isdir)=>{var node;try{node=FS.lookupNode(dir,name);}catch(e){return e.errno}var errCode=FS.nodePermissions(dir,"wx");if(errCode){return errCode}if(isdir){if(!FS.isDir(node.mode)){return 54}if(FS.isRoot(node)||FS.getPath(node)===FS.cwd()){return 10}}else {if(FS.isDir(node.mode)){return 31}}return 0},mayOpen:(node,flags)=>{if(!node){return 44}if(FS.isLink(node.mode)){return 32}else if(FS.isDir(node.mode)){if(FS.flagsToPermissionString(flags)!=="r"||flags&512){return 31}}return FS.nodePermissions(node,FS.flagsToPermissionString(flags))},MAX_OPEN_FDS:4096,nextfd:()=>{for(var fd=0;fd<=FS.MAX_OPEN_FDS;fd++){if(!FS.streams[fd]){return fd}}throw new FS.ErrnoError(33)},getStreamChecked:fd=>{var stream=FS.getStream(fd);if(!stream){throw new FS.ErrnoError(8)}return stream},getStream:fd=>FS.streams[fd],createStream:(stream,fd=-1)=>{if(!FS.FSStream){FS.FSStream=function(){this.shared={};};FS.FSStream.prototype={};Object.defineProperties(FS.FSStream.prototype,{object:{get(){return this.node},set(val){this.node=val;}},isRead:{get(){return (this.flags&2097155)!==1}},isWrite:{get(){return (this.flags&2097155)!==0}},isAppend:{get(){return this.flags&1024}},flags:{get(){return this.shared.flags},set(val){this.shared.flags=val;}},position:{get(){return this.shared.position},set(val){this.shared.position=val;}}});}stream=Object.assign(new FS.FSStream,stream);if(fd==-1){fd=FS.nextfd();}stream.fd=fd;FS.streams[fd]=stream;return stream},closeStream:fd=>{FS.streams[fd]=null;},chrdev_stream_ops:{open:stream=>{var device=FS.getDevice(stream.node.rdev);stream.stream_ops=device.stream_ops;if(stream.stream_ops.open){stream.stream_ops.open(stream);}},llseek:()=>{throw new FS.ErrnoError(70)}},major:dev=>dev>>8,minor:dev=>dev&255,makedev:(ma,mi)=>ma<<8|mi,registerDevice:(dev,ops)=>{FS.devices[dev]={stream_ops:ops};},getDevice:dev=>FS.devices[dev],getMounts:mount=>{var mounts=[];var check=[mount];while(check.length){var m=check.pop();mounts.push(m);check.push.apply(check,m.mounts);}return mounts},syncfs:(populate,callback)=>{if(typeof populate=="function"){callback=populate;populate=false;}FS.syncFSRequests++;if(FS.syncFSRequests>1){err(`warning: ${FS.syncFSRequests} FS.syncfs operations in flight at once, probably just doing extra work`);}var mounts=FS.getMounts(FS.root.mount);var completed=0;function doCallback(errCode){FS.syncFSRequests--;return callback(errCode)}function done(errCode){if(errCode){if(!done.errored){done.errored=true;return doCallback(errCode)}return}if(++completed>=mounts.length){doCallback(null);}}mounts.forEach((mount=>{if(!mount.type.syncfs){return done(null)}mount.type.syncfs(mount,populate,done);}));},mount:(type,opts,mountpoint)=>{var root=mountpoint==="/";var pseudo=!mountpoint;var node;if(root&&FS.root){throw new FS.ErrnoError(10)}else if(!root&&!pseudo){var lookup=FS.lookupPath(mountpoint,{follow_mount:false});mountpoint=lookup.path;node=lookup.node;if(FS.isMountpoint(node)){throw new FS.ErrnoError(10)}if(!FS.isDir(node.mode)){throw new FS.ErrnoError(54)}}var mount={type:type,opts:opts,mountpoint:mountpoint,mounts:[]};var mountRoot=type.mount(mount);mountRoot.mount=mount;mount.root=mountRoot;if(root){FS.root=mountRoot;}else if(node){node.mounted=mount;if(node.mount){node.mount.mounts.push(mount);}}return mountRoot},unmount:mountpoint=>{var lookup=FS.lookupPath(mountpoint,{follow_mount:false});if(!FS.isMountpoint(lookup.node)){throw new FS.ErrnoError(28)}var node=lookup.node;var mount=node.mounted;var mounts=FS.getMounts(mount);Object.keys(FS.nameTable).forEach((hash=>{var current=FS.nameTable[hash];while(current){var next=current.name_next;if(mounts.includes(current.mount)){FS.destroyNode(current);}current=next;}}));node.mounted=null;var idx=node.mount.mounts.indexOf(mount);node.mount.mounts.splice(idx,1);},lookup:(parent,name)=>parent.node_ops.lookup(parent,name),mknod:(path,mode,dev)=>{var lookup=FS.lookupPath(path,{parent:true});var parent=lookup.node;var name=PATH.basename(path);if(!name||name==="."||name===".."){throw new FS.ErrnoError(28)}var errCode=FS.mayCreate(parent,name);if(errCode){throw new FS.ErrnoError(errCode)}if(!parent.node_ops.mknod){throw new FS.ErrnoError(63)}return parent.node_ops.mknod(parent,name,mode,dev)},create:(path,mode)=>{mode=mode!==undefined?mode:438;mode&=4095;mode|=32768;return FS.mknod(path,mode,0)},mkdir:(path,mode)=>{mode=mode!==undefined?mode:511;mode&=511|512;mode|=16384;return FS.mknod(path,mode,0)},mkdirTree:(path,mode)=>{var dirs=path.split("/");var d="";for(var i=0;i<dirs.length;++i){if(!dirs[i])continue;d+="/"+dirs[i];try{FS.mkdir(d,mode);}catch(e){if(e.errno!=20)throw e}}},mkdev:(path,mode,dev)=>{if(typeof dev=="undefined"){dev=mode;mode=438;}mode|=8192;return FS.mknod(path,mode,dev)},symlink:(oldpath,newpath)=>{if(!PATH_FS.resolve(oldpath)){throw new FS.ErrnoError(44)}var lookup=FS.lookupPath(newpath,{parent:true});var parent=lookup.node;if(!parent){throw new FS.ErrnoError(44)}var newname=PATH.basename(newpath);var errCode=FS.mayCreate(parent,newname);if(errCode){throw new FS.ErrnoError(errCode)}if(!parent.node_ops.symlink){throw new FS.ErrnoError(63)}return parent.node_ops.symlink(parent,newname,oldpath)},rename:(old_path,new_path)=>{var old_dirname=PATH.dirname(old_path);var new_dirname=PATH.dirname(new_path);var old_name=PATH.basename(old_path);var new_name=PATH.basename(new_path);var lookup,old_dir,new_dir;lookup=FS.lookupPath(old_path,{parent:true});old_dir=lookup.node;lookup=FS.lookupPath(new_path,{parent:true});new_dir=lookup.node;if(!old_dir||!new_dir)throw new FS.ErrnoError(44);if(old_dir.mount!==new_dir.mount){throw new FS.ErrnoError(75)}var old_node=FS.lookupNode(old_dir,old_name);var relative=PATH_FS.relative(old_path,new_dirname);if(relative.charAt(0)!=="."){throw new FS.ErrnoError(28)}relative=PATH_FS.relative(new_path,old_dirname);if(relative.charAt(0)!=="."){throw new FS.ErrnoError(55)}var new_node;try{new_node=FS.lookupNode(new_dir,new_name);}catch(e){}if(old_node===new_node){return}var isdir=FS.isDir(old_node.mode);var errCode=FS.mayDelete(old_dir,old_name,isdir);if(errCode){throw new FS.ErrnoError(errCode)}errCode=new_node?FS.mayDelete(new_dir,new_name,isdir):FS.mayCreate(new_dir,new_name);if(errCode){throw new FS.ErrnoError(errCode)}if(!old_dir.node_ops.rename){throw new FS.ErrnoError(63)}if(FS.isMountpoint(old_node)||new_node&&FS.isMountpoint(new_node)){throw new FS.ErrnoError(10)}if(new_dir!==old_dir){errCode=FS.nodePermissions(old_dir,"w");if(errCode){throw new FS.ErrnoError(errCode)}}FS.hashRemoveNode(old_node);try{old_dir.node_ops.rename(old_node,new_dir,new_name);}catch(e){throw e}finally{FS.hashAddNode(old_node);}},rmdir:path=>{var lookup=FS.lookupPath(path,{parent:true});var parent=lookup.node;var name=PATH.basename(path);var node=FS.lookupNode(parent,name);var errCode=FS.mayDelete(parent,name,true);if(errCode){throw new FS.ErrnoError(errCode)}if(!parent.node_ops.rmdir){throw new FS.ErrnoError(63)}if(FS.isMountpoint(node)){throw new FS.ErrnoError(10)}parent.node_ops.rmdir(parent,name);FS.destroyNode(node);},readdir:path=>{var lookup=FS.lookupPath(path,{follow:true});var node=lookup.node;if(!node.node_ops.readdir){throw new FS.ErrnoError(54)}return node.node_ops.readdir(node)},unlink:path=>{var lookup=FS.lookupPath(path,{parent:true});var parent=lookup.node;if(!parent){throw new FS.ErrnoError(44)}var name=PATH.basename(path);var node=FS.lookupNode(parent,name);var errCode=FS.mayDelete(parent,name,false);if(errCode){throw new FS.ErrnoError(errCode)}if(!parent.node_ops.unlink){throw new FS.ErrnoError(63)}if(FS.isMountpoint(node)){throw new FS.ErrnoError(10)}parent.node_ops.unlink(parent,name);FS.destroyNode(node);},readlink:path=>{var lookup=FS.lookupPath(path);var link=lookup.node;if(!link){throw new FS.ErrnoError(44)}if(!link.node_ops.readlink){throw new FS.ErrnoError(28)}return PATH_FS.resolve(FS.getPath(link.parent),link.node_ops.readlink(link))},stat:(path,dontFollow)=>{var lookup=FS.lookupPath(path,{follow:!dontFollow});var node=lookup.node;if(!node){throw new FS.ErrnoError(44)}if(!node.node_ops.getattr){throw new FS.ErrnoError(63)}return node.node_ops.getattr(node)},lstat:path=>FS.stat(path,true),chmod:(path,mode,dontFollow)=>{var node;if(typeof path=="string"){var lookup=FS.lookupPath(path,{follow:!dontFollow});node=lookup.node;}else {node=path;}if(!node.node_ops.setattr){throw new FS.ErrnoError(63)}node.node_ops.setattr(node,{mode:mode&4095|node.mode&~4095,timestamp:Date.now()});},lchmod:(path,mode)=>{FS.chmod(path,mode,true);},fchmod:(fd,mode)=>{var stream=FS.getStreamChecked(fd);FS.chmod(stream.node,mode);},chown:(path,uid,gid,dontFollow)=>{var node;if(typeof path=="string"){var lookup=FS.lookupPath(path,{follow:!dontFollow});node=lookup.node;}else {node=path;}if(!node.node_ops.setattr){throw new FS.ErrnoError(63)}node.node_ops.setattr(node,{timestamp:Date.now()});},lchown:(path,uid,gid)=>{FS.chown(path,uid,gid,true);},fchown:(fd,uid,gid)=>{var stream=FS.getStreamChecked(fd);FS.chown(stream.node,uid,gid);},truncate:(path,len)=>{if(len<0){throw new FS.ErrnoError(28)}var node;if(typeof path=="string"){var lookup=FS.lookupPath(path,{follow:true});node=lookup.node;}else {node=path;}if(!node.node_ops.setattr){throw new FS.ErrnoError(63)}if(FS.isDir(node.mode)){throw new FS.ErrnoError(31)}if(!FS.isFile(node.mode)){throw new FS.ErrnoError(28)}var errCode=FS.nodePermissions(node,"w");if(errCode){throw new FS.ErrnoError(errCode)}node.node_ops.setattr(node,{size:len,timestamp:Date.now()});},ftruncate:(fd,len)=>{var stream=FS.getStreamChecked(fd);if((stream.flags&2097155)===0){throw new FS.ErrnoError(28)}FS.truncate(stream.node,len);},utime:(path,atime,mtime)=>{var lookup=FS.lookupPath(path,{follow:true});var node=lookup.node;node.node_ops.setattr(node,{timestamp:Math.max(atime,mtime)});},open:(path,flags,mode)=>{if(path===""){throw new FS.ErrnoError(44)}flags=typeof flags=="string"?FS_modeStringToFlags(flags):flags;mode=typeof mode=="undefined"?438:mode;if(flags&64){mode=mode&4095|32768;}else {mode=0;}var node;if(typeof path=="object"){node=path;}else {path=PATH.normalize(path);try{var lookup=FS.lookupPath(path,{follow:!(flags&131072)});node=lookup.node;}catch(e){}}var created=false;if(flags&64){if(node){if(flags&128){throw new FS.ErrnoError(20)}}else {node=FS.mknod(path,mode,0);created=true;}}if(!node){throw new FS.ErrnoError(44)}if(FS.isChrdev(node.mode)){flags&=~512;}if(flags&65536&&!FS.isDir(node.mode)){throw new FS.ErrnoError(54)}if(!created){var errCode=FS.mayOpen(node,flags);if(errCode){throw new FS.ErrnoError(errCode)}}if(flags&512&&!created){FS.truncate(node,0);}flags&=~(128|512|131072);var stream=FS.createStream({node:node,path:FS.getPath(node),flags:flags,seekable:true,position:0,stream_ops:node.stream_ops,ungotten:[],error:false});if(stream.stream_ops.open){stream.stream_ops.open(stream);}if(Module["logReadFiles"]&&!(flags&1)){if(!FS.readFiles)FS.readFiles={};if(!(path in FS.readFiles)){FS.readFiles[path]=1;}}return stream},close:stream=>{if(FS.isClosed(stream)){throw new FS.ErrnoError(8)}if(stream.getdents)stream.getdents=null;try{if(stream.stream_ops.close){stream.stream_ops.close(stream);}}catch(e){throw e}finally{FS.closeStream(stream.fd);}stream.fd=null;},isClosed:stream=>stream.fd===null,llseek:(stream,offset,whence)=>{if(FS.isClosed(stream)){throw new FS.ErrnoError(8)}if(!stream.seekable||!stream.stream_ops.llseek){throw new FS.ErrnoError(70)}if(whence!=0&&whence!=1&&whence!=2){throw new FS.ErrnoError(28)}stream.position=stream.stream_ops.llseek(stream,offset,whence);stream.ungotten=[];return stream.position},read:(stream,buffer,offset,length,position)=>{if(length<0||position<0){throw new FS.ErrnoError(28)}if(FS.isClosed(stream)){throw new FS.ErrnoError(8)}if((stream.flags&2097155)===1){throw new FS.ErrnoError(8)}if(FS.isDir(stream.node.mode)){throw new FS.ErrnoError(31)}if(!stream.stream_ops.read){throw new FS.ErrnoError(28)}var seeking=typeof position!="undefined";if(!seeking){position=stream.position;}else if(!stream.seekable){throw new FS.ErrnoError(70)}var bytesRead=stream.stream_ops.read(stream,buffer,offset,length,position);if(!seeking)stream.position+=bytesRead;return bytesRead},write:(stream,buffer,offset,length,position,canOwn)=>{if(length<0||position<0){throw new FS.ErrnoError(28)}if(FS.isClosed(stream)){throw new FS.ErrnoError(8)}if((stream.flags&2097155)===0){throw new FS.ErrnoError(8)}if(FS.isDir(stream.node.mode)){throw new FS.ErrnoError(31)}if(!stream.stream_ops.write){throw new FS.ErrnoError(28)}if(stream.seekable&&stream.flags&1024){FS.llseek(stream,0,2);}var seeking=typeof position!="undefined";if(!seeking){position=stream.position;}else if(!stream.seekable){throw new FS.ErrnoError(70)}var bytesWritten=stream.stream_ops.write(stream,buffer,offset,length,position,canOwn);if(!seeking)stream.position+=bytesWritten;return bytesWritten},allocate:(stream,offset,length)=>{if(FS.isClosed(stream)){throw new FS.ErrnoError(8)}if(offset<0||length<=0){throw new FS.ErrnoError(28)}if((stream.flags&2097155)===0){throw new FS.ErrnoError(8)}if(!FS.isFile(stream.node.mode)&&!FS.isDir(stream.node.mode)){throw new FS.ErrnoError(43)}if(!stream.stream_ops.allocate){throw new FS.ErrnoError(138)}stream.stream_ops.allocate(stream,offset,length);},mmap:(stream,length,position,prot,flags)=>{if((prot&2)!==0&&(flags&2)===0&&(stream.flags&2097155)!==2){throw new FS.ErrnoError(2)}if((stream.flags&2097155)===1){throw new FS.ErrnoError(2)}if(!stream.stream_ops.mmap){throw new FS.ErrnoError(43)}return stream.stream_ops.mmap(stream,length,position,prot,flags)},msync:(stream,buffer,offset,length,mmapFlags)=>{if(!stream.stream_ops.msync){return 0}return stream.stream_ops.msync(stream,buffer,offset,length,mmapFlags)},munmap:stream=>0,ioctl:(stream,cmd,arg)=>{if(!stream.stream_ops.ioctl){throw new FS.ErrnoError(59)}return stream.stream_ops.ioctl(stream,cmd,arg)},readFile:(path,opts={})=>{opts.flags=opts.flags||0;opts.encoding=opts.encoding||"binary";if(opts.encoding!=="utf8"&&opts.encoding!=="binary"){throw new Error(`Invalid encoding type "${opts.encoding}"`)}var ret;var stream=FS.open(path,opts.flags);var stat=FS.stat(path);var length=stat.size;var buf=new Uint8Array(length);FS.read(stream,buf,0,length,0);if(opts.encoding==="utf8"){ret=UTF8ArrayToString(buf,0);}else if(opts.encoding==="binary"){ret=buf;}FS.close(stream);return ret},writeFile:(path,data,opts={})=>{opts.flags=opts.flags||577;var stream=FS.open(path,opts.flags,opts.mode);if(typeof data=="string"){var buf=new Uint8Array(lengthBytesUTF8(data)+1);var actualNumBytes=stringToUTF8Array(data,buf,0,buf.length);FS.write(stream,buf,0,actualNumBytes,undefined,opts.canOwn);}else if(ArrayBuffer.isView(data)){FS.write(stream,data,0,data.byteLength,undefined,opts.canOwn);}else {throw new Error("Unsupported data type")}FS.close(stream);},cwd:()=>FS.currentPath,chdir:path=>{var lookup=FS.lookupPath(path,{follow:true});if(lookup.node===null){throw new FS.ErrnoError(44)}if(!FS.isDir(lookup.node.mode)){throw new FS.ErrnoError(54)}var errCode=FS.nodePermissions(lookup.node,"x");if(errCode){throw new FS.ErrnoError(errCode)}FS.currentPath=lookup.path;},createDefaultDirectories:()=>{FS.mkdir("/tmp");FS.mkdir("/home");FS.mkdir("/home/web_user");},createDefaultDevices:()=>{FS.mkdir("/dev");FS.registerDevice(FS.makedev(1,3),{read:()=>0,write:(stream,buffer,offset,length,pos)=>length});FS.mkdev("/dev/null",FS.makedev(1,3));TTY.register(FS.makedev(5,0),TTY.default_tty_ops);TTY.register(FS.makedev(6,0),TTY.default_tty1_ops);FS.mkdev("/dev/tty",FS.makedev(5,0));FS.mkdev("/dev/tty1",FS.makedev(6,0));var randomBuffer=new Uint8Array(1024),randomLeft=0;var randomByte=()=>{if(randomLeft===0){randomLeft=randomFill(randomBuffer).byteLength;}return randomBuffer[--randomLeft]};FS.createDevice("/dev","random",randomByte);FS.createDevice("/dev","urandom",randomByte);FS.mkdir("/dev/shm");FS.mkdir("/dev/shm/tmp");},createSpecialDirectories:()=>{FS.mkdir("/proc");var proc_self=FS.mkdir("/proc/self");FS.mkdir("/proc/self/fd");FS.mount({mount:()=>{var node=FS.createNode(proc_self,"fd",16384|511,73);node.node_ops={lookup:(parent,name)=>{var fd=+name;var stream=FS.getStreamChecked(fd);var ret={parent:null,mount:{mountpoint:"fake"},node_ops:{readlink:()=>stream.path}};ret.parent=ret;return ret}};return node}},{},"/proc/self/fd");},createStandardStreams:()=>{if(Module["stdin"]){FS.createDevice("/dev","stdin",Module["stdin"]);}else {FS.symlink("/dev/tty","/dev/stdin");}if(Module["stdout"]){FS.createDevice("/dev","stdout",null,Module["stdout"]);}else {FS.symlink("/dev/tty","/dev/stdout");}if(Module["stderr"]){FS.createDevice("/dev","stderr",null,Module["stderr"]);}else {FS.symlink("/dev/tty1","/dev/stderr");}FS.open("/dev/stdin",0);FS.open("/dev/stdout",1);FS.open("/dev/stderr",1);},ensureErrnoError:()=>{if(FS.ErrnoError)return;FS.ErrnoError=function ErrnoError(errno,node){this.name="ErrnoError";this.node=node;this.setErrno=function(errno){this.errno=errno;};this.setErrno(errno);this.message="FS error";};FS.ErrnoError.prototype=new Error;FS.ErrnoError.prototype.constructor=FS.ErrnoError;[44].forEach((code=>{FS.genericErrors[code]=new FS.ErrnoError(code);FS.genericErrors[code].stack="<generic error, no stack>";}));},staticInit:()=>{FS.ensureErrnoError();FS.nameTable=new Array(4096);FS.mount(MEMFS,{},"/");FS.createDefaultDirectories();FS.createDefaultDevices();FS.createSpecialDirectories();FS.filesystems={"MEMFS":MEMFS};},init:(input,output,error)=>{FS.init.initialized=true;FS.ensureErrnoError();Module["stdin"]=input||Module["stdin"];Module["stdout"]=output||Module["stdout"];Module["stderr"]=error||Module["stderr"];FS.createStandardStreams();},quit:()=>{FS.init.initialized=false;for(var i=0;i<FS.streams.length;i++){var stream=FS.streams[i];if(!stream){continue}FS.close(stream);}},findObject:(path,dontResolveLastLink)=>{var ret=FS.analyzePath(path,dontResolveLastLink);if(!ret.exists){return null}return ret.object},analyzePath:(path,dontResolveLastLink)=>{try{var lookup=FS.lookupPath(path,{follow:!dontResolveLastLink});path=lookup.path;}catch(e){}var ret={isRoot:false,exists:false,error:0,name:null,path:null,object:null,parentExists:false,parentPath:null,parentObject:null};try{var lookup=FS.lookupPath(path,{parent:true});ret.parentExists=true;ret.parentPath=lookup.path;ret.parentObject=lookup.node;ret.name=PATH.basename(path);lookup=FS.lookupPath(path,{follow:!dontResolveLastLink});ret.exists=true;ret.path=lookup.path;ret.object=lookup.node;ret.name=lookup.node.name;ret.isRoot=lookup.path==="/";}catch(e){ret.error=e.errno;}return ret},createPath:(parent,path,canRead,canWrite)=>{parent=typeof parent=="string"?parent:FS.getPath(parent);var parts=path.split("/").reverse();while(parts.length){var part=parts.pop();if(!part)continue;var current=PATH.join2(parent,part);try{FS.mkdir(current);}catch(e){}parent=current;}return current},createFile:(parent,name,properties,canRead,canWrite)=>{var path=PATH.join2(typeof parent=="string"?parent:FS.getPath(parent),name);var mode=FS_getMode(canRead,canWrite);return FS.create(path,mode)},createDataFile:(parent,name,data,canRead,canWrite,canOwn)=>{var path=name;if(parent){parent=typeof parent=="string"?parent:FS.getPath(parent);path=name?PATH.join2(parent,name):parent;}var mode=FS_getMode(canRead,canWrite);var node=FS.create(path,mode);if(data){if(typeof data=="string"){var arr=new Array(data.length);for(var i=0,len=data.length;i<len;++i)arr[i]=data.charCodeAt(i);data=arr;}FS.chmod(node,mode|146);var stream=FS.open(node,577);FS.write(stream,data,0,data.length,0,canOwn);FS.close(stream);FS.chmod(node,mode);}return node},createDevice:(parent,name,input,output)=>{var path=PATH.join2(typeof parent=="string"?parent:FS.getPath(parent),name);var mode=FS_getMode(!!input,!!output);if(!FS.createDevice.major)FS.createDevice.major=64;var dev=FS.makedev(FS.createDevice.major++,0);FS.registerDevice(dev,{open:stream=>{stream.seekable=false;},close:stream=>{if(output&&output.buffer&&output.buffer.length){output(10);}},read:(stream,buffer,offset,length,pos)=>{var bytesRead=0;for(var i=0;i<length;i++){var result;try{result=input();}catch(e){throw new FS.ErrnoError(29)}if(result===undefined&&bytesRead===0){throw new FS.ErrnoError(6)}if(result===null||result===undefined)break;bytesRead++;buffer[offset+i]=result;}if(bytesRead){stream.node.timestamp=Date.now();}return bytesRead},write:(stream,buffer,offset,length,pos)=>{for(var i=0;i<length;i++){try{output(buffer[offset+i]);}catch(e){throw new FS.ErrnoError(29)}}if(length){stream.node.timestamp=Date.now();}return i}});return FS.mkdev(path,mode,dev)},forceLoadFile:obj=>{if(obj.isDevice||obj.isFolder||obj.link||obj.contents)return true;if(typeof XMLHttpRequest!="undefined"){throw new Error("Lazy loading should have been performed (contents set) in createLazyFile, but it was not. Lazy loading only works in web workers. Use --embed-file or --preload-file in emcc on the main thread.")}else if(read_){try{obj.contents=intArrayFromString(read_(obj.url),true);obj.usedBytes=obj.contents.length;}catch(e){throw new FS.ErrnoError(29)}}else {throw new Error("Cannot load without read() or XMLHttpRequest.")}},createLazyFile:(parent,name,url,canRead,canWrite)=>{function LazyUint8Array(){this.lengthKnown=false;this.chunks=[];}LazyUint8Array.prototype.get=function LazyUint8Array_get(idx){if(idx>this.length-1||idx<0){return undefined}var chunkOffset=idx%this.chunkSize;var chunkNum=idx/this.chunkSize|0;return this.getter(chunkNum)[chunkOffset]};LazyUint8Array.prototype.setDataGetter=function LazyUint8Array_setDataGetter(getter){this.getter=getter;};LazyUint8Array.prototype.cacheLength=function LazyUint8Array_cacheLength(){var xhr=new XMLHttpRequest;xhr.open("HEAD",url,false);xhr.send(null);if(!(xhr.status>=200&&xhr.status<300||xhr.status===304))throw new Error("Couldn't load "+url+". Status: "+xhr.status);var datalength=Number(xhr.getResponseHeader("Content-length"));var header;var hasByteServing=(header=xhr.getResponseHeader("Accept-Ranges"))&&header==="bytes";var usesGzip=(header=xhr.getResponseHeader("Content-Encoding"))&&header==="gzip";var chunkSize=1024*1024;if(!hasByteServing)chunkSize=datalength;var doXHR=(from,to)=>{if(from>to)throw new Error("invalid range ("+from+", "+to+") or no bytes requested!");if(to>datalength-1)throw new Error("only "+datalength+" bytes available! programmer error!");var xhr=new XMLHttpRequest;xhr.open("GET",url,false);if(datalength!==chunkSize)xhr.setRequestHeader("Range","bytes="+from+"-"+to);xhr.responseType="arraybuffer";if(xhr.overrideMimeType){xhr.overrideMimeType("text/plain; charset=x-user-defined");}xhr.send(null);if(!(xhr.status>=200&&xhr.status<300||xhr.status===304))throw new Error("Couldn't load "+url+". Status: "+xhr.status);if(xhr.response!==undefined){return new Uint8Array(xhr.response||[])}return intArrayFromString(xhr.responseText||"",true)};var lazyArray=this;lazyArray.setDataGetter((chunkNum=>{var start=chunkNum*chunkSize;var end=(chunkNum+1)*chunkSize-1;end=Math.min(end,datalength-1);if(typeof lazyArray.chunks[chunkNum]=="undefined"){lazyArray.chunks[chunkNum]=doXHR(start,end);}if(typeof lazyArray.chunks[chunkNum]=="undefined")throw new Error("doXHR failed!");return lazyArray.chunks[chunkNum]}));if(usesGzip||!datalength){chunkSize=datalength=1;datalength=this.getter(0).length;chunkSize=datalength;out("LazyFiles on gzip forces download of the whole file when length is accessed");}this._length=datalength;this._chunkSize=chunkSize;this.lengthKnown=true;};if(typeof XMLHttpRequest!="undefined"){if(!ENVIRONMENT_IS_WORKER)throw "Cannot do synchronous binary XHRs outside webworkers in modern browsers. Use --embed-file or --preload-file in emcc";var lazyArray=new LazyUint8Array;Object.defineProperties(lazyArray,{length:{get:function(){if(!this.lengthKnown){this.cacheLength();}return this._length}},chunkSize:{get:function(){if(!this.lengthKnown){this.cacheLength();}return this._chunkSize}}});var properties={isDevice:false,contents:lazyArray};}else {var properties={isDevice:false,url:url};}var node=FS.createFile(parent,name,properties,canRead,canWrite);if(properties.contents){node.contents=properties.contents;}else if(properties.url){node.contents=null;node.url=properties.url;}Object.defineProperties(node,{usedBytes:{get:function(){return this.contents.length}}});var stream_ops={};var keys=Object.keys(node.stream_ops);keys.forEach((key=>{var fn=node.stream_ops[key];stream_ops[key]=function forceLoadLazyFile(){FS.forceLoadFile(node);return fn.apply(null,arguments)};}));function writeChunks(stream,buffer,offset,length,position){var contents=stream.node.contents;if(position>=contents.length)return 0;var size=Math.min(contents.length-position,length);if(contents.slice){for(var i=0;i<size;i++){buffer[offset+i]=contents[position+i];}}else {for(var i=0;i<size;i++){buffer[offset+i]=contents.get(position+i);}}return size}stream_ops.read=(stream,buffer,offset,length,position)=>{FS.forceLoadFile(node);return writeChunks(stream,buffer,offset,length,position)};stream_ops.mmap=(stream,length,position,prot,flags)=>{FS.forceLoadFile(node);var ptr=mmapAlloc();if(!ptr){throw new FS.ErrnoError(48)}writeChunks(stream,HEAP8,ptr,length,position);return {ptr:ptr,allocated:true}};node.stream_ops=stream_ops;return node}};var UTF8ToString=(ptr,maxBytesToRead)=>ptr?UTF8ArrayToString(HEAPU8,ptr,maxBytesToRead):"";var SYSCALLS={DEFAULT_POLLMASK:5,calculateAt:function(dirfd,path,allowEmpty){if(PATH.isAbs(path)){return path}var dir;if(dirfd===-100){dir=FS.cwd();}else {var dirstream=SYSCALLS.getStreamFromFD(dirfd);dir=dirstream.path;}if(path.length==0){if(!allowEmpty){throw new FS.ErrnoError(44)}return dir}return PATH.join2(dir,path)},doStat:function(func,path,buf){try{var stat=func(path);}catch(e){if(e&&e.node&&PATH.normalize(path)!==PATH.normalize(FS.getPath(e.node))){return -54}throw e}HEAP32[buf>>2]=stat.dev;HEAP32[buf+4>>2]=stat.mode;HEAPU32[buf+8>>2]=stat.nlink;HEAP32[buf+12>>2]=stat.uid;HEAP32[buf+16>>2]=stat.gid;HEAP32[buf+20>>2]=stat.rdev;tempI64=[stat.size>>>0,(tempDouble=stat.size,+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)],HEAP32[buf+24>>2]=tempI64[0],HEAP32[buf+28>>2]=tempI64[1];HEAP32[buf+32>>2]=4096;HEAP32[buf+36>>2]=stat.blocks;var atime=stat.atime.getTime();var mtime=stat.mtime.getTime();var ctime=stat.ctime.getTime();tempI64=[Math.floor(atime/1e3)>>>0,(tempDouble=Math.floor(atime/1e3),+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)],HEAP32[buf+40>>2]=tempI64[0],HEAP32[buf+44>>2]=tempI64[1];HEAPU32[buf+48>>2]=atime%1e3*1e3;tempI64=[Math.floor(mtime/1e3)>>>0,(tempDouble=Math.floor(mtime/1e3),+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)],HEAP32[buf+56>>2]=tempI64[0],HEAP32[buf+60>>2]=tempI64[1];HEAPU32[buf+64>>2]=mtime%1e3*1e3;tempI64=[Math.floor(ctime/1e3)>>>0,(tempDouble=Math.floor(ctime/1e3),+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)],HEAP32[buf+72>>2]=tempI64[0],HEAP32[buf+76>>2]=tempI64[1];HEAPU32[buf+80>>2]=ctime%1e3*1e3;tempI64=[stat.ino>>>0,(tempDouble=stat.ino,+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)],HEAP32[buf+88>>2]=tempI64[0],HEAP32[buf+92>>2]=tempI64[1];return 0},doMsync:function(addr,stream,len,flags,offset){if(!FS.isFile(stream.node.mode)){throw new FS.ErrnoError(43)}if(flags&2){return 0}var buffer=HEAPU8.slice(addr,addr+len);FS.msync(stream,buffer,offset,len,flags);},varargs:undefined,get(){SYSCALLS.varargs+=4;var ret=HEAP32[SYSCALLS.varargs-4>>2];return ret},getStr(ptr){var ret=UTF8ToString(ptr);return ret},getStreamFromFD:function(fd){var stream=FS.getStreamChecked(fd);return stream}};function ___syscall_dup3(fd,newfd,flags){try{var old=SYSCALLS.getStreamFromFD(fd);if(old.fd===newfd)return -28;var existing=FS.getStream(newfd);if(existing)FS.close(existing);return FS.createStream(old,newfd).fd}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}var setErrNo=value=>{HEAP32[___errno_location()>>2]=value;return value};function ___syscall_fcntl64(fd,cmd,varargs){SYSCALLS.varargs=varargs;try{var stream=SYSCALLS.getStreamFromFD(fd);switch(cmd){case 0:{var arg=SYSCALLS.get();if(arg<0){return -28}var newStream;newStream=FS.createStream(stream,arg);return newStream.fd}case 1:case 2:return 0;case 3:return stream.flags;case 4:{var arg=SYSCALLS.get();stream.flags|=arg;return 0}case 5:{var arg=SYSCALLS.get();var offset=0;HEAP16[arg+offset>>1]=2;return 0}case 6:case 7:return 0;case 16:case 8:return -28;case 9:setErrNo(28);return -1;default:{return -28}}}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}function ___syscall_ioctl(fd,op,varargs){SYSCALLS.varargs=varargs;try{var stream=SYSCALLS.getStreamFromFD(fd);switch(op){case 21509:{if(!stream.tty)return -59;return 0}case 21505:{if(!stream.tty)return -59;if(stream.tty.ops.ioctl_tcgets){var termios=stream.tty.ops.ioctl_tcgets(stream);var argp=SYSCALLS.get();HEAP32[argp>>2]=termios.c_iflag||0;HEAP32[argp+4>>2]=termios.c_oflag||0;HEAP32[argp+8>>2]=termios.c_cflag||0;HEAP32[argp+12>>2]=termios.c_lflag||0;for(var i=0;i<32;i++){HEAP8[argp+i+17>>0]=termios.c_cc[i]||0;}return 0}return 0}case 21510:case 21511:case 21512:{if(!stream.tty)return -59;return 0}case 21506:case 21507:case 21508:{if(!stream.tty)return -59;if(stream.tty.ops.ioctl_tcsets){var argp=SYSCALLS.get();var c_iflag=HEAP32[argp>>2];var c_oflag=HEAP32[argp+4>>2];var c_cflag=HEAP32[argp+8>>2];var c_lflag=HEAP32[argp+12>>2];var c_cc=[];for(var i=0;i<32;i++){c_cc.push(HEAP8[argp+i+17>>0]);}return stream.tty.ops.ioctl_tcsets(stream.tty,op,{c_iflag:c_iflag,c_oflag:c_oflag,c_cflag:c_cflag,c_lflag:c_lflag,c_cc:c_cc})}return 0}case 21519:{if(!stream.tty)return -59;var argp=SYSCALLS.get();HEAP32[argp>>2]=0;return 0}case 21520:{if(!stream.tty)return -59;return -28}case 21531:{var argp=SYSCALLS.get();return FS.ioctl(stream,op,argp)}case 21523:{if(!stream.tty)return -59;if(stream.tty.ops.ioctl_tiocgwinsz){var winsize=stream.tty.ops.ioctl_tiocgwinsz(stream.tty);var argp=SYSCALLS.get();HEAP16[argp>>1]=winsize[0];HEAP16[argp+2>>1]=winsize[1];}return 0}case 21524:{if(!stream.tty)return -59;return 0}case 21515:{if(!stream.tty)return -59;return 0}default:return -28}}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}function ___syscall_lstat64(path,buf){try{path=SYSCALLS.getStr(path);return SYSCALLS.doStat(FS.lstat,path,buf)}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}function ___syscall_openat(dirfd,path,flags,varargs){SYSCALLS.varargs=varargs;try{path=SYSCALLS.getStr(path);path=SYSCALLS.calculateAt(dirfd,path);var mode=varargs?SYSCALLS.get():0;return FS.open(path,flags,mode).fd}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}function ___syscall_renameat(olddirfd,oldpath,newdirfd,newpath){try{oldpath=SYSCALLS.getStr(oldpath);newpath=SYSCALLS.getStr(newpath);oldpath=SYSCALLS.calculateAt(olddirfd,oldpath);newpath=SYSCALLS.calculateAt(newdirfd,newpath);FS.rename(oldpath,newpath);return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}function ___syscall_rmdir(path){try{path=SYSCALLS.getStr(path);FS.rmdir(path);return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}function ___syscall_unlinkat(dirfd,path,flags){try{path=SYSCALLS.getStr(path);path=SYSCALLS.calculateAt(dirfd,path);if(flags===0){FS.unlink(path);}else if(flags===512){FS.rmdir(path);}else {abort("Invalid flags passed to unlinkat");}return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return -e.errno}}var nowIsMonotonic=true;var __emscripten_get_now_is_monotonic=()=>nowIsMonotonic;var __emscripten_throw_longjmp=()=>{throw Infinity};function convertI32PairToI53Checked(lo,hi){return hi+2097152>>>0<4194305-!!lo?(lo>>>0)+hi*4294967296:NaN}function __gmtime_js(time_low,time_high,tmPtr){var time=convertI32PairToI53Checked(time_low,time_high);var date=new Date(time*1e3);HEAP32[tmPtr>>2]=date.getUTCSeconds();HEAP32[tmPtr+4>>2]=date.getUTCMinutes();HEAP32[tmPtr+8>>2]=date.getUTCHours();HEAP32[tmPtr+12>>2]=date.getUTCDate();HEAP32[tmPtr+16>>2]=date.getUTCMonth();HEAP32[tmPtr+20>>2]=date.getUTCFullYear()-1900;HEAP32[tmPtr+24>>2]=date.getUTCDay();var start=Date.UTC(date.getUTCFullYear(),0,1,0,0,0,0);var yday=(date.getTime()-start)/(1e3*60*60*24)|0;HEAP32[tmPtr+28>>2]=yday;}var isLeapYear=year=>year%4===0&&(year%100!==0||year%400===0);var MONTH_DAYS_LEAP_CUMULATIVE=[0,31,60,91,121,152,182,213,244,274,305,335];var MONTH_DAYS_REGULAR_CUMULATIVE=[0,31,59,90,120,151,181,212,243,273,304,334];var ydayFromDate=date=>{var leap=isLeapYear(date.getFullYear());var monthDaysCumulative=leap?MONTH_DAYS_LEAP_CUMULATIVE:MONTH_DAYS_REGULAR_CUMULATIVE;var yday=monthDaysCumulative[date.getMonth()]+date.getDate()-1;return yday};function __localtime_js(time_low,time_high,tmPtr){var time=convertI32PairToI53Checked(time_low,time_high);var date=new Date(time*1e3);HEAP32[tmPtr>>2]=date.getSeconds();HEAP32[tmPtr+4>>2]=date.getMinutes();HEAP32[tmPtr+8>>2]=date.getHours();HEAP32[tmPtr+12>>2]=date.getDate();HEAP32[tmPtr+16>>2]=date.getMonth();HEAP32[tmPtr+20>>2]=date.getFullYear()-1900;HEAP32[tmPtr+24>>2]=date.getDay();var yday=ydayFromDate(date)|0;HEAP32[tmPtr+28>>2]=yday;HEAP32[tmPtr+36>>2]=-(date.getTimezoneOffset()*60);var start=new Date(date.getFullYear(),0,1);var summerOffset=new Date(date.getFullYear(),6,1).getTimezoneOffset();var winterOffset=start.getTimezoneOffset();var dst=(summerOffset!=winterOffset&&date.getTimezoneOffset()==Math.min(winterOffset,summerOffset))|0;HEAP32[tmPtr+32>>2]=dst;}var __mktime_js=function(tmPtr){var ret=(()=>{var date=new Date(HEAP32[tmPtr+20>>2]+1900,HEAP32[tmPtr+16>>2],HEAP32[tmPtr+12>>2],HEAP32[tmPtr+8>>2],HEAP32[tmPtr+4>>2],HEAP32[tmPtr>>2],0);var dst=HEAP32[tmPtr+32>>2];var guessedOffset=date.getTimezoneOffset();var start=new Date(date.getFullYear(),0,1);var summerOffset=new Date(date.getFullYear(),6,1).getTimezoneOffset();var winterOffset=start.getTimezoneOffset();var dstOffset=Math.min(winterOffset,summerOffset);if(dst<0){HEAP32[tmPtr+32>>2]=Number(summerOffset!=winterOffset&&dstOffset==guessedOffset);}else if(dst>0!=(dstOffset==guessedOffset)){var nonDstOffset=Math.max(winterOffset,summerOffset);var trueOffset=dst>0?dstOffset:nonDstOffset;date.setTime(date.getTime()+(trueOffset-guessedOffset)*6e4);}HEAP32[tmPtr+24>>2]=date.getDay();var yday=ydayFromDate(date)|0;HEAP32[tmPtr+28>>2]=yday;HEAP32[tmPtr>>2]=date.getSeconds();HEAP32[tmPtr+4>>2]=date.getMinutes();HEAP32[tmPtr+8>>2]=date.getHours();HEAP32[tmPtr+12>>2]=date.getDate();HEAP32[tmPtr+16>>2]=date.getMonth();HEAP32[tmPtr+20>>2]=date.getYear();return date.getTime()/1e3})();return setTempRet0((tempDouble=ret,+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)),ret>>>0};var stringToUTF8=(str,outPtr,maxBytesToWrite)=>stringToUTF8Array(str,HEAPU8,outPtr,maxBytesToWrite);var stringToNewUTF8=str=>{var size=lengthBytesUTF8(str)+1;var ret=_malloc(size);if(ret)stringToUTF8(str,ret,size);return ret};var __tzset_js=(timezone,daylight,tzname)=>{var currentYear=(new Date).getFullYear();var winter=new Date(currentYear,0,1);var summer=new Date(currentYear,6,1);var winterOffset=winter.getTimezoneOffset();var summerOffset=summer.getTimezoneOffset();var stdTimezoneOffset=Math.max(winterOffset,summerOffset);HEAPU32[timezone>>2]=stdTimezoneOffset*60;HEAP32[daylight>>2]=Number(winterOffset!=summerOffset);function extractZone(date){var match=date.toTimeString().match(/\(([A-Za-z ]+)\)$/);return match?match[1]:"GMT"}var winterName=extractZone(winter);var summerName=extractZone(summer);var winterNamePtr=stringToNewUTF8(winterName);var summerNamePtr=stringToNewUTF8(summerName);if(summerOffset<winterOffset){HEAPU32[tzname>>2]=winterNamePtr;HEAPU32[tzname+4>>2]=summerNamePtr;}else {HEAPU32[tzname>>2]=summerNamePtr;HEAPU32[tzname+4>>2]=winterNamePtr;}};var _abort=()=>{abort("");};function _emscripten_date_now(){return Date.now()}var _emscripten_memcpy_big=(dest,src,num)=>HEAPU8.copyWithin(dest,src,src+num);var getHeapMax=()=>2147483648;var growMemory=size=>{var b=wasmMemory.buffer;var pages=size-b.byteLength+65535>>>16;try{wasmMemory.grow(pages);updateMemoryViews();return 1}catch(e){}};var _emscripten_resize_heap=requestedSize=>{var oldSize=HEAPU8.length;requestedSize>>>=0;var maxHeapSize=getHeapMax();if(requestedSize>maxHeapSize){return false}var alignUp=(x,multiple)=>x+(multiple-x%multiple)%multiple;for(var cutDown=1;cutDown<=4;cutDown*=2){var overGrownHeapSize=oldSize*(1+.2/cutDown);overGrownHeapSize=Math.min(overGrownHeapSize,requestedSize+100663296);var newSize=Math.min(maxHeapSize,alignUp(Math.max(requestedSize,overGrownHeapSize),65536));var replacement=growMemory(newSize);if(replacement){return true}}return false};var ENV={};var getExecutableName=()=>thisProgram||"./this.program";var getEnvStrings=()=>{if(!getEnvStrings.strings){var lang=(typeof navigator=="object"&&navigator.languages&&navigator.languages[0]||"C").replace("-","_")+".UTF-8";var env={"USER":"web_user","LOGNAME":"web_user","PATH":"/","PWD":"/","HOME":"/home/web_user","LANG":lang,"_":getExecutableName()};for(var x in ENV){if(ENV[x]===undefined)delete env[x];else env[x]=ENV[x];}var strings=[];for(var x in env){strings.push(`${x}=${env[x]}`);}getEnvStrings.strings=strings;}return getEnvStrings.strings};var stringToAscii=(str,buffer)=>{for(var i=0;i<str.length;++i){HEAP8[buffer++>>0]=str.charCodeAt(i);}HEAP8[buffer>>0]=0;};var _environ_get=(__environ,environ_buf)=>{var bufSize=0;getEnvStrings().forEach((function(string,i){var ptr=environ_buf+bufSize;HEAPU32[__environ+i*4>>2]=ptr;stringToAscii(string,ptr);bufSize+=string.length+1;}));return 0};var _environ_sizes_get=(penviron_count,penviron_buf_size)=>{var strings=getEnvStrings();HEAPU32[penviron_count>>2]=strings.length;var bufSize=0;strings.forEach((function(string){bufSize+=string.length+1;}));HEAPU32[penviron_buf_size>>2]=bufSize;return 0};var _proc_exit=code=>{if(!keepRuntimeAlive()){if(Module["onExit"])Module["onExit"](code);ABORT=true;}quit_(code,new ExitStatus(code));};var exitJS=(status,implicit)=>{_proc_exit(status);};var _exit=exitJS;function _fd_close(fd){try{var stream=SYSCALLS.getStreamFromFD(fd);FS.close(stream);return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return e.errno}}var doReadv=(stream,iov,iovcnt,offset)=>{var ret=0;for(var i=0;i<iovcnt;i++){var ptr=HEAPU32[iov>>2];var len=HEAPU32[iov+4>>2];iov+=8;var curr=FS.read(stream,HEAP8,ptr,len,offset);if(curr<0)return -1;ret+=curr;if(curr<len)break;if(typeof offset!=="undefined"){offset+=curr;}}return ret};function _fd_read(fd,iov,iovcnt,pnum){try{var stream=SYSCALLS.getStreamFromFD(fd);var num=doReadv(stream,iov,iovcnt);HEAPU32[pnum>>2]=num;return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return e.errno}}function _fd_seek(fd,offset_low,offset_high,whence,newOffset){var offset=convertI32PairToI53Checked(offset_low,offset_high);try{if(isNaN(offset))return 61;var stream=SYSCALLS.getStreamFromFD(fd);FS.llseek(stream,offset,whence);tempI64=[stream.position>>>0,(tempDouble=stream.position,+Math.abs(tempDouble)>=1?tempDouble>0?+Math.floor(tempDouble/4294967296)>>>0:~~+Math.ceil((tempDouble-+(~~tempDouble>>>0))/4294967296)>>>0:0)],HEAP32[newOffset>>2]=tempI64[0],HEAP32[newOffset+4>>2]=tempI64[1];if(stream.getdents&&offset===0&&whence===0)stream.getdents=null;return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return e.errno}}var doWritev=(stream,iov,iovcnt,offset)=>{var ret=0;for(var i=0;i<iovcnt;i++){var ptr=HEAPU32[iov>>2];var len=HEAPU32[iov+4>>2];iov+=8;var curr=FS.write(stream,HEAP8,ptr,len,offset);if(curr<0)return -1;ret+=curr;if(typeof offset!=="undefined"){offset+=curr;}}return ret};function _fd_write(fd,iov,iovcnt,pnum){try{var stream=SYSCALLS.getStreamFromFD(fd);var num=doWritev(stream,iov,iovcnt);HEAPU32[pnum>>2]=num;return 0}catch(e){if(typeof FS=="undefined"||!(e.name==="ErrnoError"))throw e;return e.errno}}var arraySum=(array,index)=>{var sum=0;for(var i=0;i<=index;sum+=array[i++]){}return sum};var MONTH_DAYS_LEAP=[31,29,31,30,31,30,31,31,30,31,30,31];var MONTH_DAYS_REGULAR=[31,28,31,30,31,30,31,31,30,31,30,31];var addDays=(date,days)=>{var newDate=new Date(date.getTime());while(days>0){var leap=isLeapYear(newDate.getFullYear());var currentMonth=newDate.getMonth();var daysInCurrentMonth=(leap?MONTH_DAYS_LEAP:MONTH_DAYS_REGULAR)[currentMonth];if(days>daysInCurrentMonth-newDate.getDate()){days-=daysInCurrentMonth-newDate.getDate()+1;newDate.setDate(1);if(currentMonth<11){newDate.setMonth(currentMonth+1);}else {newDate.setMonth(0);newDate.setFullYear(newDate.getFullYear()+1);}}else {newDate.setDate(newDate.getDate()+days);return newDate}}return newDate};var writeArrayToMemory=(array,buffer)=>{HEAP8.set(array,buffer);};var _strftime=(s,maxsize,format,tm)=>{var tm_zone=HEAP32[tm+40>>2];var date={tm_sec:HEAP32[tm>>2],tm_min:HEAP32[tm+4>>2],tm_hour:HEAP32[tm+8>>2],tm_mday:HEAP32[tm+12>>2],tm_mon:HEAP32[tm+16>>2],tm_year:HEAP32[tm+20>>2],tm_wday:HEAP32[tm+24>>2],tm_yday:HEAP32[tm+28>>2],tm_isdst:HEAP32[tm+32>>2],tm_gmtoff:HEAP32[tm+36>>2],tm_zone:tm_zone?UTF8ToString(tm_zone):""};var pattern=UTF8ToString(format);var EXPANSION_RULES_1={"%c":"%a %b %d %H:%M:%S %Y","%D":"%m/%d/%y","%F":"%Y-%m-%d","%h":"%b","%r":"%I:%M:%S %p","%R":"%H:%M","%T":"%H:%M:%S","%x":"%m/%d/%y","%X":"%H:%M:%S","%Ec":"%c","%EC":"%C","%Ex":"%m/%d/%y","%EX":"%H:%M:%S","%Ey":"%y","%EY":"%Y","%Od":"%d","%Oe":"%e","%OH":"%H","%OI":"%I","%Om":"%m","%OM":"%M","%OS":"%S","%Ou":"%u","%OU":"%U","%OV":"%V","%Ow":"%w","%OW":"%W","%Oy":"%y"};for(var rule in EXPANSION_RULES_1){pattern=pattern.replace(new RegExp(rule,"g"),EXPANSION_RULES_1[rule]);}var WEEKDAYS=["Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"];var MONTHS=["January","February","March","April","May","June","July","August","September","October","November","December"];function leadingSomething(value,digits,character){var str=typeof value=="number"?value.toString():value||"";while(str.length<digits){str=character[0]+str;}return str}function leadingNulls(value,digits){return leadingSomething(value,digits,"0")}function compareByDay(date1,date2){function sgn(value){return value<0?-1:value>0?1:0}var compare;if((compare=sgn(date1.getFullYear()-date2.getFullYear()))===0){if((compare=sgn(date1.getMonth()-date2.getMonth()))===0){compare=sgn(date1.getDate()-date2.getDate());}}return compare}function getFirstWeekStartDate(janFourth){switch(janFourth.getDay()){case 0:return new Date(janFourth.getFullYear()-1,11,29);case 1:return janFourth;case 2:return new Date(janFourth.getFullYear(),0,3);case 3:return new Date(janFourth.getFullYear(),0,2);case 4:return new Date(janFourth.getFullYear(),0,1);case 5:return new Date(janFourth.getFullYear()-1,11,31);case 6:return new Date(janFourth.getFullYear()-1,11,30)}}function getWeekBasedYear(date){var thisDate=addDays(new Date(date.tm_year+1900,0,1),date.tm_yday);var janFourthThisYear=new Date(thisDate.getFullYear(),0,4);var janFourthNextYear=new Date(thisDate.getFullYear()+1,0,4);var firstWeekStartThisYear=getFirstWeekStartDate(janFourthThisYear);var firstWeekStartNextYear=getFirstWeekStartDate(janFourthNextYear);if(compareByDay(firstWeekStartThisYear,thisDate)<=0){if(compareByDay(firstWeekStartNextYear,thisDate)<=0){return thisDate.getFullYear()+1}return thisDate.getFullYear()}return thisDate.getFullYear()-1}var EXPANSION_RULES_2={"%a":date=>WEEKDAYS[date.tm_wday].substring(0,3),"%A":date=>WEEKDAYS[date.tm_wday],"%b":date=>MONTHS[date.tm_mon].substring(0,3),"%B":date=>MONTHS[date.tm_mon],"%C":date=>{var year=date.tm_year+1900;return leadingNulls(year/100|0,2)},"%d":date=>leadingNulls(date.tm_mday,2),"%e":date=>leadingSomething(date.tm_mday,2," "),"%g":date=>getWeekBasedYear(date).toString().substring(2),"%G":date=>getWeekBasedYear(date),"%H":date=>leadingNulls(date.tm_hour,2),"%I":date=>{var twelveHour=date.tm_hour;if(twelveHour==0)twelveHour=12;else if(twelveHour>12)twelveHour-=12;return leadingNulls(twelveHour,2)},"%j":date=>leadingNulls(date.tm_mday+arraySum(isLeapYear(date.tm_year+1900)?MONTH_DAYS_LEAP:MONTH_DAYS_REGULAR,date.tm_mon-1),3),"%m":date=>leadingNulls(date.tm_mon+1,2),"%M":date=>leadingNulls(date.tm_min,2),"%n":()=>"\n","%p":date=>{if(date.tm_hour>=0&&date.tm_hour<12){return "AM"}return "PM"},"%S":date=>leadingNulls(date.tm_sec,2),"%t":()=>"\t","%u":date=>date.tm_wday||7,"%U":date=>{var days=date.tm_yday+7-date.tm_wday;return leadingNulls(Math.floor(days/7),2)},"%V":date=>{var val=Math.floor((date.tm_yday+7-(date.tm_wday+6)%7)/7);if((date.tm_wday+371-date.tm_yday-2)%7<=2){val++;}if(!val){val=52;var dec31=(date.tm_wday+7-date.tm_yday-1)%7;if(dec31==4||dec31==5&&isLeapYear(date.tm_year%400-1)){val++;}}else if(val==53){var jan1=(date.tm_wday+371-date.tm_yday)%7;if(jan1!=4&&(jan1!=3||!isLeapYear(date.tm_year)))val=1;}return leadingNulls(val,2)},"%w":date=>date.tm_wday,"%W":date=>{var days=date.tm_yday+7-(date.tm_wday+6)%7;return leadingNulls(Math.floor(days/7),2)},"%y":date=>(date.tm_year+1900).toString().substring(2),"%Y":date=>date.tm_year+1900,"%z":date=>{var off=date.tm_gmtoff;var ahead=off>=0;off=Math.abs(off)/60;off=off/60*100+off%60;return (ahead?"+":"-")+String("0000"+off).slice(-4)},"%Z":date=>date.tm_zone,"%%":()=>"%"};pattern=pattern.replace(/%%/g,"\0\0");for(var rule in EXPANSION_RULES_2){if(pattern.includes(rule)){pattern=pattern.replace(new RegExp(rule,"g"),EXPANSION_RULES_2[rule](date));}}pattern=pattern.replace(/\0\0/g,"%");var bytes=intArrayFromString(pattern,false);if(bytes.length>maxsize){return 0}writeArrayToMemory(bytes,s);return bytes.length-1};var _system=command=>{if(ENVIRONMENT_IS_NODE){if(!command)return 1;var cmdstr=UTF8ToString(command);if(!cmdstr.length)return 0;var cp=require$1("child_process");var ret=cp.spawnSync(cmdstr,[],{shell:true,stdio:"inherit"});var _W_EXITCODE=(ret,sig)=>ret<<8|sig;if(ret.status===null){var signalToNumber=sig=>{switch(sig){case"SIGHUP":return 1;case"SIGINT":return 2;case"SIGQUIT":return 3;case"SIGFPE":return 8;case"SIGKILL":return 9;case"SIGALRM":return 14;case"SIGTERM":return 15}return 2};return _W_EXITCODE(0,signalToNumber(ret.signal))}return _W_EXITCODE(ret.status,0)}if(!command)return 0;setErrNo(52);return -1};var wasmTableMirror=[];var getWasmTableEntry=funcPtr=>{var func=wasmTableMirror[funcPtr];if(!func){if(funcPtr>=wasmTableMirror.length)wasmTableMirror.length=funcPtr+1;wasmTableMirror[funcPtr]=func=wasmTable.get(funcPtr);}return func};function getCFunc(ident){var func=Module["_"+ident];return func}var stringToUTF8OnStack=str=>{var size=lengthBytesUTF8(str)+1;var ret=stackAlloc(size);stringToUTF8(str,ret,size);return ret};var ccall=function(ident,returnType,argTypes,args,opts){var toC={"string":str=>{var ret=0;if(str!==null&&str!==undefined&&str!==0){ret=stringToUTF8OnStack(str);}return ret},"array":arr=>{var ret=stackAlloc(arr.length);writeArrayToMemory(arr,ret);return ret}};function convertReturnValue(ret){if(returnType==="string"){return UTF8ToString(ret)}if(returnType==="boolean")return Boolean(ret);return ret}var func=getCFunc(ident);var cArgs=[];var stack=0;if(args){for(var i=0;i<args.length;i++){var converter=toC[argTypes[i]];if(converter){if(stack===0)stack=stackSave();cArgs[i]=converter(args[i]);}else {cArgs[i]=args[i];}}}var ret=func.apply(null,cArgs);function onDone(ret){if(stack!==0)stackRestore(stack);return convertReturnValue(ret)}ret=onDone(ret);return ret};function uleb128Encode(n,target){if(n<128){target.push(n);}else {target.push(n%128|128,n>>7);}}function sigToWasmTypes(sig){var typeNames={"i":"i32","j":"i64","f":"f32","d":"f64","p":"i32"};var type={parameters:[],results:sig[0]=="v"?[]:[typeNames[sig[0]]]};for(var i=1;i<sig.length;++i){type.parameters.push(typeNames[sig[i]]);}return type}function generateFuncType(sig,target){var sigRet=sig.slice(0,1);var sigParam=sig.slice(1);var typeCodes={"i":127,"p":127,"j":126,"f":125,"d":124};target.push(96);uleb128Encode(sigParam.length,target);for(var i=0;i<sigParam.length;++i){target.push(typeCodes[sigParam[i]]);}if(sigRet=="v"){target.push(0);}else {target.push(1,typeCodes[sigRet]);}}function convertJsFunctionToWasm(func,sig){if(typeof WebAssembly.Function=="function"){return new WebAssembly.Function(sigToWasmTypes(sig),func)}var typeSectionBody=[1];generateFuncType(sig,typeSectionBody);var bytes=[0,97,115,109,1,0,0,0,1];uleb128Encode(typeSectionBody.length,bytes);bytes.push.apply(bytes,typeSectionBody);bytes.push(2,7,1,1,101,1,102,0,0,7,5,1,1,102,0,0);var module=new WebAssembly.Module(new Uint8Array(bytes));var instance=new WebAssembly.Instance(module,{"e":{"f":func}});var wrappedFunc=instance.exports["f"];return wrappedFunc}function updateTableMap(offset,count){if(functionsInTableMap){for(var i=offset;i<offset+count;i++){var item=getWasmTableEntry(i);if(item){functionsInTableMap.set(item,i);}}}}var functionsInTableMap=undefined;function getFunctionAddress(func){if(!functionsInTableMap){functionsInTableMap=new WeakMap;updateTableMap(0,wasmTable.length);}return functionsInTableMap.get(func)||0}var freeTableIndexes=[];function getEmptyTableSlot(){if(freeTableIndexes.length){return freeTableIndexes.pop()}try{wasmTable.grow(1);}catch(err){if(!(err instanceof RangeError)){throw err}throw "Unable to grow wasm table. Set ALLOW_TABLE_GROWTH."}return wasmTable.length-1}var setWasmTableEntry=(idx,func)=>{wasmTable.set(idx,func);wasmTableMirror[idx]=wasmTable.get(idx);};function addFunction(func,sig){var rtn=getFunctionAddress(func);if(rtn){return rtn}var ret=getEmptyTableSlot();try{setWasmTableEntry(ret,func);}catch(err){if(!(err instanceof TypeError)){throw err}var wrapped=convertJsFunctionToWasm(func,sig);setWasmTableEntry(ret,wrapped);}functionsInTableMap.set(func,ret);return ret}function removeFunction(index){functionsInTableMap.delete(getWasmTableEntry(index));freeTableIndexes.push(index);}var FSNode=function(parent,name,mode,rdev){if(!parent){parent=this;}this.parent=parent;this.mount=parent.mount;this.mounted=null;this.id=FS.nextInode++;this.name=name;this.mode=mode;this.node_ops={};this.stream_ops={};this.rdev=rdev;};var readMode=292|73;var writeMode=146;Object.defineProperties(FSNode.prototype,{read:{get:function(){return (this.mode&readMode)===readMode},set:function(val){val?this.mode|=readMode:this.mode&=~readMode;}},write:{get:function(){return (this.mode&writeMode)===writeMode},set:function(val){val?this.mode|=writeMode:this.mode&=~writeMode;}},isFolder:{get:function(){return FS.isDir(this.mode)}},isDevice:{get:function(){return FS.isChrdev(this.mode)}}});FS.FSNode=FSNode;FS.createPreloadedFile=FS_createPreloadedFile;FS.staticInit();var wasmImports={x:___syscall_dup3,b:___syscall_fcntl64,z:___syscall_ioctl,r:___syscall_lstat64,d:___syscall_openat,t:___syscall_renameat,u:___syscall_rmdir,e:___syscall_unlinkat,g:__emscripten_get_now_is_monotonic,o:__emscripten_throw_longjmp,k:__gmtime_js,l:__localtime_js,m:__mktime_js,q:__tzset_js,B:_abort,a:_emscripten_date_now,h:_emscripten_memcpy_big,p:_emscripten_resize_heap,v:_environ_get,w:_environ_sizes_get,i:_exit,c:_fd_close,y:_fd_read,n:_fd_seek,f:_fd_write,A:invoke_vii,s:_strftime,j:_system};createWasm();Module["_lua_checkstack"]=function(){return (Module["_lua_checkstack"]=Module["asm"]["E"]).apply(null,arguments)};Module["_lua_xmove"]=function(){return (Module["_lua_xmove"]=Module["asm"]["F"]).apply(null,arguments)};Module["_lua_atpanic"]=function(){return (Module["_lua_atpanic"]=Module["asm"]["G"]).apply(null,arguments)};Module["_lua_version"]=function(){return (Module["_lua_version"]=Module["asm"]["H"]).apply(null,arguments)};Module["_lua_absindex"]=function(){return (Module["_lua_absindex"]=Module["asm"]["I"]).apply(null,arguments)};Module["_lua_gettop"]=function(){return (Module["_lua_gettop"]=Module["asm"]["J"]).apply(null,arguments)};Module["_lua_settop"]=function(){return (Module["_lua_settop"]=Module["asm"]["K"]).apply(null,arguments)};Module["_lua_closeslot"]=function(){return (Module["_lua_closeslot"]=Module["asm"]["L"]).apply(null,arguments)};Module["_lua_rotate"]=function(){return (Module["_lua_rotate"]=Module["asm"]["M"]).apply(null,arguments)};Module["_lua_copy"]=function(){return (Module["_lua_copy"]=Module["asm"]["N"]).apply(null,arguments)};Module["_lua_pushvalue"]=function(){return (Module["_lua_pushvalue"]=Module["asm"]["O"]).apply(null,arguments)};Module["_lua_type"]=function(){return (Module["_lua_type"]=Module["asm"]["P"]).apply(null,arguments)};Module["_lua_typename"]=function(){return (Module["_lua_typename"]=Module["asm"]["Q"]).apply(null,arguments)};Module["_lua_iscfunction"]=function(){return (Module["_lua_iscfunction"]=Module["asm"]["R"]).apply(null,arguments)};Module["_lua_isinteger"]=function(){return (Module["_lua_isinteger"]=Module["asm"]["S"]).apply(null,arguments)};Module["_lua_isnumber"]=function(){return (Module["_lua_isnumber"]=Module["asm"]["T"]).apply(null,arguments)};Module["_lua_isstring"]=function(){return (Module["_lua_isstring"]=Module["asm"]["U"]).apply(null,arguments)};Module["_lua_isuserdata"]=function(){return (Module["_lua_isuserdata"]=Module["asm"]["V"]).apply(null,arguments)};Module["_lua_rawequal"]=function(){return (Module["_lua_rawequal"]=Module["asm"]["W"]).apply(null,arguments)};Module["_lua_arith"]=function(){return (Module["_lua_arith"]=Module["asm"]["X"]).apply(null,arguments)};Module["_lua_compare"]=function(){return (Module["_lua_compare"]=Module["asm"]["Y"]).apply(null,arguments)};Module["_lua_stringtonumber"]=function(){return (Module["_lua_stringtonumber"]=Module["asm"]["Z"]).apply(null,arguments)};Module["_lua_tonumberx"]=function(){return (Module["_lua_tonumberx"]=Module["asm"]["_"]).apply(null,arguments)};Module["_lua_tointegerx"]=function(){return (Module["_lua_tointegerx"]=Module["asm"]["$"]).apply(null,arguments)};Module["_lua_toboolean"]=function(){return (Module["_lua_toboolean"]=Module["asm"]["aa"]).apply(null,arguments)};Module["_lua_tolstring"]=function(){return (Module["_lua_tolstring"]=Module["asm"]["ba"]).apply(null,arguments)};Module["_lua_rawlen"]=function(){return (Module["_lua_rawlen"]=Module["asm"]["ca"]).apply(null,arguments)};Module["_lua_tocfunction"]=function(){return (Module["_lua_tocfunction"]=Module["asm"]["da"]).apply(null,arguments)};Module["_lua_touserdata"]=function(){return (Module["_lua_touserdata"]=Module["asm"]["ea"]).apply(null,arguments)};Module["_lua_tothread"]=function(){return (Module["_lua_tothread"]=Module["asm"]["fa"]).apply(null,arguments)};Module["_lua_topointer"]=function(){return (Module["_lua_topointer"]=Module["asm"]["ga"]).apply(null,arguments)};Module["_lua_pushnil"]=function(){return (Module["_lua_pushnil"]=Module["asm"]["ha"]).apply(null,arguments)};Module["_lua_pushnumber"]=function(){return (Module["_lua_pushnumber"]=Module["asm"]["ia"]).apply(null,arguments)};Module["_lua_pushinteger"]=function(){return (Module["_lua_pushinteger"]=Module["asm"]["ja"]).apply(null,arguments)};Module["_lua_pushlstring"]=function(){return (Module["_lua_pushlstring"]=Module["asm"]["ka"]).apply(null,arguments)};Module["_lua_pushstring"]=function(){return (Module["_lua_pushstring"]=Module["asm"]["la"]).apply(null,arguments)};Module["_lua_pushcclosure"]=function(){return (Module["_lua_pushcclosure"]=Module["asm"]["ma"]).apply(null,arguments)};Module["_lua_pushboolean"]=function(){return (Module["_lua_pushboolean"]=Module["asm"]["na"]).apply(null,arguments)};Module["_lua_pushlightuserdata"]=function(){return (Module["_lua_pushlightuserdata"]=Module["asm"]["oa"]).apply(null,arguments)};Module["_lua_pushthread"]=function(){return (Module["_lua_pushthread"]=Module["asm"]["pa"]).apply(null,arguments)};Module["_lua_getglobal"]=function(){return (Module["_lua_getglobal"]=Module["asm"]["qa"]).apply(null,arguments)};Module["_lua_gettable"]=function(){return (Module["_lua_gettable"]=Module["asm"]["ra"]).apply(null,arguments)};Module["_lua_getfield"]=function(){return (Module["_lua_getfield"]=Module["asm"]["sa"]).apply(null,arguments)};Module["_lua_geti"]=function(){return (Module["_lua_geti"]=Module["asm"]["ta"]).apply(null,arguments)};Module["_lua_rawget"]=function(){return (Module["_lua_rawget"]=Module["asm"]["ua"]).apply(null,arguments)};Module["_lua_rawgeti"]=function(){return (Module["_lua_rawgeti"]=Module["asm"]["va"]).apply(null,arguments)};Module["_lua_rawgetp"]=function(){return (Module["_lua_rawgetp"]=Module["asm"]["wa"]).apply(null,arguments)};Module["_lua_createtable"]=function(){return (Module["_lua_createtable"]=Module["asm"]["xa"]).apply(null,arguments)};Module["_lua_getmetatable"]=function(){return (Module["_lua_getmetatable"]=Module["asm"]["ya"]).apply(null,arguments)};Module["_lua_getiuservalue"]=function(){return (Module["_lua_getiuservalue"]=Module["asm"]["za"]).apply(null,arguments)};Module["_lua_setglobal"]=function(){return (Module["_lua_setglobal"]=Module["asm"]["Aa"]).apply(null,arguments)};Module["_lua_settable"]=function(){return (Module["_lua_settable"]=Module["asm"]["Ba"]).apply(null,arguments)};Module["_lua_setfield"]=function(){return (Module["_lua_setfield"]=Module["asm"]["Ca"]).apply(null,arguments)};Module["_lua_seti"]=function(){return (Module["_lua_seti"]=Module["asm"]["Da"]).apply(null,arguments)};Module["_lua_rawset"]=function(){return (Module["_lua_rawset"]=Module["asm"]["Ea"]).apply(null,arguments)};Module["_lua_rawsetp"]=function(){return (Module["_lua_rawsetp"]=Module["asm"]["Fa"]).apply(null,arguments)};Module["_lua_rawseti"]=function(){return (Module["_lua_rawseti"]=Module["asm"]["Ga"]).apply(null,arguments)};Module["_lua_setmetatable"]=function(){return (Module["_lua_setmetatable"]=Module["asm"]["Ha"]).apply(null,arguments)};Module["_lua_setiuservalue"]=function(){return (Module["_lua_setiuservalue"]=Module["asm"]["Ia"]).apply(null,arguments)};Module["_lua_callk"]=function(){return (Module["_lua_callk"]=Module["asm"]["Ja"]).apply(null,arguments)};Module["_lua_pcallk"]=function(){return (Module["_lua_pcallk"]=Module["asm"]["Ka"]).apply(null,arguments)};Module["_lua_load"]=function(){return (Module["_lua_load"]=Module["asm"]["La"]).apply(null,arguments)};Module["_lua_dump"]=function(){return (Module["_lua_dump"]=Module["asm"]["Ma"]).apply(null,arguments)};Module["_lua_status"]=function(){return (Module["_lua_status"]=Module["asm"]["Na"]).apply(null,arguments)};Module["_lua_error"]=function(){return (Module["_lua_error"]=Module["asm"]["Oa"]).apply(null,arguments)};Module["_lua_next"]=function(){return (Module["_lua_next"]=Module["asm"]["Pa"]).apply(null,arguments)};Module["_lua_toclose"]=function(){return (Module["_lua_toclose"]=Module["asm"]["Qa"]).apply(null,arguments)};Module["_lua_concat"]=function(){return (Module["_lua_concat"]=Module["asm"]["Ra"]).apply(null,arguments)};Module["_lua_len"]=function(){return (Module["_lua_len"]=Module["asm"]["Sa"]).apply(null,arguments)};Module["_lua_getallocf"]=function(){return (Module["_lua_getallocf"]=Module["asm"]["Ta"]).apply(null,arguments)};Module["_lua_setallocf"]=function(){return (Module["_lua_setallocf"]=Module["asm"]["Ua"]).apply(null,arguments)};Module["_lua_setwarnf"]=function(){return (Module["_lua_setwarnf"]=Module["asm"]["Va"]).apply(null,arguments)};Module["_lua_warning"]=function(){return (Module["_lua_warning"]=Module["asm"]["Wa"]).apply(null,arguments)};Module["_lua_newuserdatauv"]=function(){return (Module["_lua_newuserdatauv"]=Module["asm"]["Xa"]).apply(null,arguments)};Module["_lua_getupvalue"]=function(){return (Module["_lua_getupvalue"]=Module["asm"]["Ya"]).apply(null,arguments)};Module["_lua_setupvalue"]=function(){return (Module["_lua_setupvalue"]=Module["asm"]["Za"]).apply(null,arguments)};Module["_lua_upvalueid"]=function(){return (Module["_lua_upvalueid"]=Module["asm"]["_a"]).apply(null,arguments)};Module["_lua_upvaluejoin"]=function(){return (Module["_lua_upvaluejoin"]=Module["asm"]["$a"]).apply(null,arguments)};Module["_luaL_traceback"]=function(){return (Module["_luaL_traceback"]=Module["asm"]["bb"]).apply(null,arguments)};Module["_lua_getstack"]=function(){return (Module["_lua_getstack"]=Module["asm"]["cb"]).apply(null,arguments)};Module["_lua_getinfo"]=function(){return (Module["_lua_getinfo"]=Module["asm"]["db"]).apply(null,arguments)};Module["_luaL_buffinit"]=function(){return (Module["_luaL_buffinit"]=Module["asm"]["eb"]).apply(null,arguments)};Module["_luaL_addstring"]=function(){return (Module["_luaL_addstring"]=Module["asm"]["fb"]).apply(null,arguments)};Module["_luaL_prepbuffsize"]=function(){return (Module["_luaL_prepbuffsize"]=Module["asm"]["gb"]).apply(null,arguments)};Module["_luaL_addvalue"]=function(){return (Module["_luaL_addvalue"]=Module["asm"]["hb"]).apply(null,arguments)};Module["_luaL_pushresult"]=function(){return (Module["_luaL_pushresult"]=Module["asm"]["ib"]).apply(null,arguments)};Module["_luaL_argerror"]=function(){return (Module["_luaL_argerror"]=Module["asm"]["jb"]).apply(null,arguments)};Module["_luaL_typeerror"]=function(){return (Module["_luaL_typeerror"]=Module["asm"]["kb"]).apply(null,arguments)};Module["_luaL_getmetafield"]=function(){return (Module["_luaL_getmetafield"]=Module["asm"]["lb"]).apply(null,arguments)};Module["_luaL_where"]=function(){return (Module["_luaL_where"]=Module["asm"]["mb"]).apply(null,arguments)};Module["_luaL_fileresult"]=function(){return (Module["_luaL_fileresult"]=Module["asm"]["nb"]).apply(null,arguments)};var ___errno_location=function(){return (___errno_location=Module["asm"]["ob"]).apply(null,arguments)};Module["_luaL_execresult"]=function(){return (Module["_luaL_execresult"]=Module["asm"]["pb"]).apply(null,arguments)};Module["_luaL_newmetatable"]=function(){return (Module["_luaL_newmetatable"]=Module["asm"]["qb"]).apply(null,arguments)};Module["_luaL_setmetatable"]=function(){return (Module["_luaL_setmetatable"]=Module["asm"]["rb"]).apply(null,arguments)};Module["_luaL_testudata"]=function(){return (Module["_luaL_testudata"]=Module["asm"]["sb"]).apply(null,arguments)};Module["_luaL_checkudata"]=function(){return (Module["_luaL_checkudata"]=Module["asm"]["tb"]).apply(null,arguments)};Module["_luaL_optlstring"]=function(){return (Module["_luaL_optlstring"]=Module["asm"]["ub"]).apply(null,arguments)};Module["_luaL_checklstring"]=function(){return (Module["_luaL_checklstring"]=Module["asm"]["vb"]).apply(null,arguments)};Module["_luaL_checkstack"]=function(){return (Module["_luaL_checkstack"]=Module["asm"]["wb"]).apply(null,arguments)};Module["_luaL_checktype"]=function(){return (Module["_luaL_checktype"]=Module["asm"]["xb"]).apply(null,arguments)};Module["_luaL_checkany"]=function(){return (Module["_luaL_checkany"]=Module["asm"]["yb"]).apply(null,arguments)};Module["_luaL_checknumber"]=function(){return (Module["_luaL_checknumber"]=Module["asm"]["zb"]).apply(null,arguments)};Module["_luaL_optnumber"]=function(){return (Module["_luaL_optnumber"]=Module["asm"]["Ab"]).apply(null,arguments)};Module["_luaL_checkinteger"]=function(){return (Module["_luaL_checkinteger"]=Module["asm"]["Bb"]).apply(null,arguments)};Module["_luaL_optinteger"]=function(){return (Module["_luaL_optinteger"]=Module["asm"]["Cb"]).apply(null,arguments)};Module["_luaL_setfuncs"]=function(){return (Module["_luaL_setfuncs"]=Module["asm"]["Db"]).apply(null,arguments)};Module["_luaL_addlstring"]=function(){return (Module["_luaL_addlstring"]=Module["asm"]["Eb"]).apply(null,arguments)};Module["_luaL_pushresultsize"]=function(){return (Module["_luaL_pushresultsize"]=Module["asm"]["Fb"]).apply(null,arguments)};Module["_luaL_buffinitsize"]=function(){return (Module["_luaL_buffinitsize"]=Module["asm"]["Gb"]).apply(null,arguments)};Module["_luaL_ref"]=function(){return (Module["_luaL_ref"]=Module["asm"]["Hb"]).apply(null,arguments)};Module["_luaL_unref"]=function(){return (Module["_luaL_unref"]=Module["asm"]["Ib"]).apply(null,arguments)};Module["_luaL_loadfilex"]=function(){return (Module["_luaL_loadfilex"]=Module["asm"]["Jb"]).apply(null,arguments)};Module["_luaL_loadbufferx"]=function(){return (Module["_luaL_loadbufferx"]=Module["asm"]["Kb"]).apply(null,arguments)};Module["_luaL_loadstring"]=function(){return (Module["_luaL_loadstring"]=Module["asm"]["Lb"]).apply(null,arguments)};Module["_luaL_callmeta"]=function(){return (Module["_luaL_callmeta"]=Module["asm"]["Mb"]).apply(null,arguments)};Module["_luaL_len"]=function(){return (Module["_luaL_len"]=Module["asm"]["Nb"]).apply(null,arguments)};Module["_luaL_tolstring"]=function(){return (Module["_luaL_tolstring"]=Module["asm"]["Ob"]).apply(null,arguments)};Module["_luaL_getsubtable"]=function(){return (Module["_luaL_getsubtable"]=Module["asm"]["Pb"]).apply(null,arguments)};Module["_luaL_requiref"]=function(){return (Module["_luaL_requiref"]=Module["asm"]["Qb"]).apply(null,arguments)};Module["_luaL_addgsub"]=function(){return (Module["_luaL_addgsub"]=Module["asm"]["Rb"]).apply(null,arguments)};Module["_luaL_gsub"]=function(){return (Module["_luaL_gsub"]=Module["asm"]["Sb"]).apply(null,arguments)};Module["_luaL_newstate"]=function(){return (Module["_luaL_newstate"]=Module["asm"]["Tb"]).apply(null,arguments)};Module["_lua_newstate"]=function(){return (Module["_lua_newstate"]=Module["asm"]["Ub"]).apply(null,arguments)};Module["_free"]=function(){return (Module["_free"]=Module["asm"]["Vb"]).apply(null,arguments)};Module["_realloc"]=function(){return (Module["_realloc"]=Module["asm"]["Wb"]).apply(null,arguments)};Module["_luaL_checkversion_"]=function(){return (Module["_luaL_checkversion_"]=Module["asm"]["Xb"]).apply(null,arguments)};Module["_luaopen_base"]=function(){return (Module["_luaopen_base"]=Module["asm"]["Yb"]).apply(null,arguments)};Module["_luaopen_coroutine"]=function(){return (Module["_luaopen_coroutine"]=Module["asm"]["Zb"]).apply(null,arguments)};Module["_lua_newthread"]=function(){return (Module["_lua_newthread"]=Module["asm"]["_b"]).apply(null,arguments)};Module["_lua_yieldk"]=function(){return (Module["_lua_yieldk"]=Module["asm"]["$b"]).apply(null,arguments)};Module["_lua_isyieldable"]=function(){return (Module["_lua_isyieldable"]=Module["asm"]["ac"]).apply(null,arguments)};Module["_lua_resetthread"]=function(){return (Module["_lua_resetthread"]=Module["asm"]["bc"]).apply(null,arguments)};Module["_lua_resume"]=function(){return (Module["_lua_resume"]=Module["asm"]["cc"]).apply(null,arguments)};Module["_luaopen_debug"]=function(){return (Module["_luaopen_debug"]=Module["asm"]["dc"]).apply(null,arguments)};Module["_lua_gethookmask"]=function(){return (Module["_lua_gethookmask"]=Module["asm"]["ec"]).apply(null,arguments)};Module["_lua_gethook"]=function(){return (Module["_lua_gethook"]=Module["asm"]["fc"]).apply(null,arguments)};Module["_lua_gethookcount"]=function(){return (Module["_lua_gethookcount"]=Module["asm"]["gc"]).apply(null,arguments)};Module["_lua_getlocal"]=function(){return (Module["_lua_getlocal"]=Module["asm"]["hc"]).apply(null,arguments)};Module["_lua_sethook"]=function(){return (Module["_lua_sethook"]=Module["asm"]["ic"]).apply(null,arguments)};Module["_lua_setlocal"]=function(){return (Module["_lua_setlocal"]=Module["asm"]["jc"]).apply(null,arguments)};Module["_lua_setcstacklimit"]=function(){return (Module["_lua_setcstacklimit"]=Module["asm"]["kc"]).apply(null,arguments)};var _malloc=Module["_malloc"]=function(){return (_malloc=Module["_malloc"]=Module["asm"]["lc"]).apply(null,arguments)};var setTempRet0=function(){return (setTempRet0=Module["asm"]["mc"]).apply(null,arguments)};Module["_luaL_openlibs"]=function(){return (Module["_luaL_openlibs"]=Module["asm"]["nc"]).apply(null,arguments)};Module["_luaopen_package"]=function(){return (Module["_luaopen_package"]=Module["asm"]["oc"]).apply(null,arguments)};Module["_luaopen_table"]=function(){return (Module["_luaopen_table"]=Module["asm"]["pc"]).apply(null,arguments)};Module["_luaopen_io"]=function(){return (Module["_luaopen_io"]=Module["asm"]["qc"]).apply(null,arguments)};Module["_luaopen_os"]=function(){return (Module["_luaopen_os"]=Module["asm"]["rc"]).apply(null,arguments)};Module["_luaopen_string"]=function(){return (Module["_luaopen_string"]=Module["asm"]["sc"]).apply(null,arguments)};Module["_luaopen_math"]=function(){return (Module["_luaopen_math"]=Module["asm"]["tc"]).apply(null,arguments)};Module["_luaopen_utf8"]=function(){return (Module["_luaopen_utf8"]=Module["asm"]["uc"]).apply(null,arguments)};Module["_lua_close"]=function(){return (Module["_lua_close"]=Module["asm"]["vc"]).apply(null,arguments)};var _setThrew=function(){return (_setThrew=Module["asm"]["wc"]).apply(null,arguments)};var stackSave=function(){return (stackSave=Module["asm"]["xc"]).apply(null,arguments)};var stackRestore=function(){return (stackRestore=Module["asm"]["yc"]).apply(null,arguments)};var stackAlloc=function(){return (stackAlloc=Module["asm"]["zc"]).apply(null,arguments)};function invoke_vii(index,a1,a2){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}Module["stackAlloc"]=stackAlloc;Module["stackSave"]=stackSave;Module["stackRestore"]=stackRestore;Module["ENV"]=ENV;Module["ccall"]=ccall;Module["addFunction"]=addFunction;Module["removeFunction"]=removeFunction;Module["setValue"]=setValue;Module["getValue"]=getValue;Module["stringToUTF8"]=stringToUTF8;Module["lengthBytesUTF8"]=lengthBytesUTF8;Module["stringToNewUTF8"]=stringToNewUTF8;Module["FS"]=FS;var calledRun;dependenciesFulfilled=function runCaller(){if(!calledRun)run();if(!calledRun)dependenciesFulfilled=runCaller;};function run(){if(runDependencies>0){return}preRun();if(runDependencies>0){return}function doRun(){if(calledRun)return;calledRun=true;Module["calledRun"]=true;if(ABORT)return;initRuntime();readyPromiseResolve(Module);if(Module["onRuntimeInitialized"])Module["onRuntimeInitialized"]();postRun();}if(Module["setStatus"]){Module["setStatus"]("Running...");setTimeout((function(){setTimeout((function(){Module["setStatus"]("");}),1);doRun();}),1);}else {doRun();}}if(Module["preInit"]){if(typeof Module["preInit"]=="function")Module["preInit"]=[Module["preInit"]];while(Module["preInit"].length>0){Module["preInit"].pop()();}}run();


      return moduleArg.ready
    }

    );
    })();

    class LuaWasm {
        static async initialize(customWasmFileLocation, environmentVariables) {
            const module = await initWasmModule({
                print: console.log,
                printErr: console.error,
                locateFile: (path, scriptDirectory) => {
                    return customWasmFileLocation || scriptDirectory + path;
                },
                preRun: (initializedModule) => {
                    if (typeof environmentVariables === 'object') {
                        Object.entries(environmentVariables).forEach(([k, v]) => (initializedModule.ENV[k] = v));
                    }
                },
            });
            return new LuaWasm(module);
        }
        constructor(module) {
            this.referenceTracker = new WeakMap();
            this.referenceMap = new Map();
            this.availableReferences = [];
            this.module = module;
            this.luaL_checkversion_ = this.cwrap('luaL_checkversion_', null, ['number', 'number', 'number']);
            this.luaL_getmetafield = this.cwrap('luaL_getmetafield', 'number', ['number', 'number', 'string']);
            this.luaL_callmeta = this.cwrap('luaL_callmeta', 'number', ['number', 'number', 'string']);
            this.luaL_tolstring = this.cwrap('luaL_tolstring', 'string', ['number', 'number', 'number']);
            this.luaL_argerror = this.cwrap('luaL_argerror', 'number', ['number', 'number', 'string']);
            this.luaL_typeerror = this.cwrap('luaL_typeerror', 'number', ['number', 'number', 'string']);
            this.luaL_checklstring = this.cwrap('luaL_checklstring', 'string', ['number', 'number', 'number']);
            this.luaL_optlstring = this.cwrap('luaL_optlstring', 'string', ['number', 'number', 'string', 'number']);
            this.luaL_checknumber = this.cwrap('luaL_checknumber', 'number', ['number', 'number']);
            this.luaL_optnumber = this.cwrap('luaL_optnumber', 'number', ['number', 'number', 'number']);
            this.luaL_checkinteger = this.cwrap('luaL_checkinteger', 'number', ['number', 'number']);
            this.luaL_optinteger = this.cwrap('luaL_optinteger', 'number', ['number', 'number', 'number']);
            this.luaL_checkstack = this.cwrap('luaL_checkstack', null, ['number', 'number', 'string']);
            this.luaL_checktype = this.cwrap('luaL_checktype', null, ['number', 'number', 'number']);
            this.luaL_checkany = this.cwrap('luaL_checkany', null, ['number', 'number']);
            this.luaL_newmetatable = this.cwrap('luaL_newmetatable', 'number', ['number', 'string']);
            this.luaL_setmetatable = this.cwrap('luaL_setmetatable', null, ['number', 'string']);
            this.luaL_testudata = this.cwrap('luaL_testudata', 'number', ['number', 'number', 'string']);
            this.luaL_checkudata = this.cwrap('luaL_checkudata', 'number', ['number', 'number', 'string']);
            this.luaL_where = this.cwrap('luaL_where', null, ['number', 'number']);
            this.luaL_fileresult = this.cwrap('luaL_fileresult', 'number', ['number', 'number', 'string']);
            this.luaL_execresult = this.cwrap('luaL_execresult', 'number', ['number', 'number']);
            this.luaL_ref = this.cwrap('luaL_ref', 'number', ['number', 'number']);
            this.luaL_unref = this.cwrap('luaL_unref', null, ['number', 'number', 'number']);
            this.luaL_loadfilex = this.cwrap('luaL_loadfilex', 'number', ['number', 'string', 'string']);
            this.luaL_loadbufferx = this.cwrap('luaL_loadbufferx', 'number', ['number', 'string|number', 'number', 'string|number', 'string']);
            this.luaL_loadstring = this.cwrap('luaL_loadstring', 'number', ['number', 'string']);
            this.luaL_newstate = this.cwrap('luaL_newstate', 'number', []);
            this.luaL_len = this.cwrap('luaL_len', 'number', ['number', 'number']);
            this.luaL_addgsub = this.cwrap('luaL_addgsub', null, ['number', 'string', 'string', 'string']);
            this.luaL_gsub = this.cwrap('luaL_gsub', 'string', ['number', 'string', 'string', 'string']);
            this.luaL_setfuncs = this.cwrap('luaL_setfuncs', null, ['number', 'number', 'number']);
            this.luaL_getsubtable = this.cwrap('luaL_getsubtable', 'number', ['number', 'number', 'string']);
            this.luaL_traceback = this.cwrap('luaL_traceback', null, ['number', 'number', 'string', 'number']);
            this.luaL_requiref = this.cwrap('luaL_requiref', null, ['number', 'string', 'number', 'number']);
            this.luaL_buffinit = this.cwrap('luaL_buffinit', null, ['number', 'number']);
            this.luaL_prepbuffsize = this.cwrap('luaL_prepbuffsize', 'string', ['number', 'number']);
            this.luaL_addlstring = this.cwrap('luaL_addlstring', null, ['number', 'string', 'number']);
            this.luaL_addstring = this.cwrap('luaL_addstring', null, ['number', 'string']);
            this.luaL_addvalue = this.cwrap('luaL_addvalue', null, ['number']);
            this.luaL_pushresult = this.cwrap('luaL_pushresult', null, ['number']);
            this.luaL_pushresultsize = this.cwrap('luaL_pushresultsize', null, ['number', 'number']);
            this.luaL_buffinitsize = this.cwrap('luaL_buffinitsize', 'string', ['number', 'number', 'number']);
            this.lua_newstate = this.cwrap('lua_newstate', 'number', ['number', 'number']);
            this.lua_close = this.cwrap('lua_close', null, ['number']);
            this.lua_newthread = this.cwrap('lua_newthread', 'number', ['number']);
            this.lua_resetthread = this.cwrap('lua_resetthread', 'number', ['number']);
            this.lua_atpanic = this.cwrap('lua_atpanic', 'number', ['number', 'number']);
            this.lua_version = this.cwrap('lua_version', 'number', ['number']);
            this.lua_absindex = this.cwrap('lua_absindex', 'number', ['number', 'number']);
            this.lua_gettop = this.cwrap('lua_gettop', 'number', ['number']);
            this.lua_settop = this.cwrap('lua_settop', null, ['number', 'number']);
            this.lua_pushvalue = this.cwrap('lua_pushvalue', null, ['number', 'number']);
            this.lua_rotate = this.cwrap('lua_rotate', null, ['number', 'number', 'number']);
            this.lua_copy = this.cwrap('lua_copy', null, ['number', 'number', 'number']);
            this.lua_checkstack = this.cwrap('lua_checkstack', 'number', ['number', 'number']);
            this.lua_xmove = this.cwrap('lua_xmove', null, ['number', 'number', 'number']);
            this.lua_isnumber = this.cwrap('lua_isnumber', 'number', ['number', 'number']);
            this.lua_isstring = this.cwrap('lua_isstring', 'number', ['number', 'number']);
            this.lua_iscfunction = this.cwrap('lua_iscfunction', 'number', ['number', 'number']);
            this.lua_isinteger = this.cwrap('lua_isinteger', 'number', ['number', 'number']);
            this.lua_isuserdata = this.cwrap('lua_isuserdata', 'number', ['number', 'number']);
            this.lua_type = this.cwrap('lua_type', 'number', ['number', 'number']);
            this.lua_typename = this.cwrap('lua_typename', 'string', ['number', 'number']);
            this.lua_tonumberx = this.cwrap('lua_tonumberx', 'number', ['number', 'number', 'number']);
            this.lua_tointegerx = this.cwrap('lua_tointegerx', 'number', ['number', 'number', 'number']);
            this.lua_toboolean = this.cwrap('lua_toboolean', 'number', ['number', 'number']);
            this.lua_tolstring = this.cwrap('lua_tolstring', 'string', ['number', 'number', 'number']);
            {
                const wrapped = this.cwrap('lua_tolstring', 'number', ['number', 'number', 'number']);
                this.lua_tou8array = (L, idx) => {
                    const stack = module.stackSave();
                    const lenPtr = module.stackAlloc(4);
                    const strPtr = wrapped(L, idx, lenPtr);
                    var len = module.getValue(lenPtr, "*");
                    module.stackRestore(stack);
                    return module.HEAPU8.slice(strPtr, strPtr + len);
                };
            }
            this.lua_rawlen = this.cwrap('lua_rawlen', 'number', ['number', 'number']);
            this.lua_tocfunction = this.cwrap('lua_tocfunction', 'number', ['number', 'number']);
            this.lua_touserdata = this.cwrap('lua_touserdata', 'number', ['number', 'number']);
            this.lua_tothread = this.cwrap('lua_tothread', 'number', ['number', 'number']);
            this.lua_topointer = this.cwrap('lua_topointer', 'number', ['number', 'number']);
            this.lua_arith = this.cwrap('lua_arith', null, ['number', 'number']);
            this.lua_rawequal = this.cwrap('lua_rawequal', 'number', ['number', 'number', 'number']);
            this.lua_compare = this.cwrap('lua_compare', 'number', ['number', 'number', 'number', 'number']);
            this.lua_pushnil = this.cwrap('lua_pushnil', null, ['number']);
            this.lua_pushnumber = this.cwrap('lua_pushnumber', null, ['number', 'number']);
            this.lua_pushinteger = this.cwrap('lua_pushinteger', null, ['number', 'number']);
            this.lua_pushlstring = this.cwrap('lua_pushlstring', 'string', ['number', 'string|number', 'number']);
            this.lua_pushstring = this.cwrap('lua_pushstring', 'string', ['number', 'string|number']);
            {
                this.lua_pushbstring = (L, str) => {
                    const buf = new Uint8Array(str.split("").map(function (c) {
                        return c.charCodeAt(0);
                    }));
                    const size = buf.length;
                    const stack = module.stackSave();
                    const bufferPointer = module.stackAlloc(size);
                    module.HEAPU8.set(buf, bufferPointer);
                    this.lua_pushlstring(L, bufferPointer, size);
                    module.stackRestore(stack);
                };
            }
            this.lua_pushcclosure = this.cwrap('lua_pushcclosure', null, ['number', 'number', 'number']);
            this.lua_pushboolean = this.cwrap('lua_pushboolean', null, ['number', 'number']);
            this.lua_pushlightuserdata = this.cwrap('lua_pushlightuserdata', null, ['number', 'number']);
            this.lua_pushthread = this.cwrap('lua_pushthread', 'number', ['number']);
            this.lua_getglobal = this.cwrap('lua_getglobal', 'number', ['number', 'string']);
            this.lua_gettable = this.cwrap('lua_gettable', 'number', ['number', 'number']);
            this.lua_getfield = this.cwrap('lua_getfield', 'number', ['number', 'number', 'string']);
            this.lua_geti = this.cwrap('lua_geti', 'number', ['number', 'number', 'number']);
            this.lua_rawget = this.cwrap('lua_rawget', 'number', ['number', 'number']);
            this.lua_rawgeti = this.cwrap('lua_rawgeti', 'number', ['number', 'number', 'number']);
            this.lua_rawgetp = this.cwrap('lua_rawgetp', 'number', ['number', 'number', 'number']);
            this.lua_createtable = this.cwrap('lua_createtable', null, ['number', 'number', 'number']);
            this.lua_newuserdatauv = this.cwrap('lua_newuserdatauv', 'number', ['number', 'number', 'number']);
            this.lua_getmetatable = this.cwrap('lua_getmetatable', 'number', ['number', 'number']);
            this.lua_getiuservalue = this.cwrap('lua_getiuservalue', 'number', ['number', 'number', 'number']);
            this.lua_setglobal = this.cwrap('lua_setglobal', null, ['number', 'string']);
            this.lua_settable = this.cwrap('lua_settable', null, ['number', 'number']);
            this.lua_setfield = this.cwrap('lua_setfield', null, ['number', 'number', 'string']);
            this.lua_seti = this.cwrap('lua_seti', null, ['number', 'number', 'number']);
            this.lua_rawset = this.cwrap('lua_rawset', null, ['number', 'number']);
            this.lua_rawseti = this.cwrap('lua_rawseti', null, ['number', 'number', 'number']);
            this.lua_rawsetp = this.cwrap('lua_rawsetp', null, ['number', 'number', 'number']);
            this.lua_setmetatable = this.cwrap('lua_setmetatable', 'number', ['number', 'number']);
            this.lua_setiuservalue = this.cwrap('lua_setiuservalue', 'number', ['number', 'number', 'number']);
            this.lua_callk = this.cwrap('lua_callk', null, ['number', 'number', 'number', 'number', 'number']);
            this.lua_pcallk = this.cwrap('lua_pcallk', 'number', ['number', 'number', 'number', 'number', 'number', 'number']);
            this.lua_load = this.cwrap('lua_load', 'number', ['number', 'number', 'number', 'string', 'string']);
            this.lua_dump = this.cwrap('lua_dump', 'number', ['number', 'number', 'number', 'number']);
            this.lua_yieldk = this.cwrap('lua_yieldk', 'number', ['number', 'number', 'number', 'number']);
            this.lua_resume = this.cwrap('lua_resume', 'number', ['number', 'number', 'number', 'number']);
            this.lua_status = this.cwrap('lua_status', 'number', ['number']);
            this.lua_isyieldable = this.cwrap('lua_isyieldable', 'number', ['number']);
            this.lua_setwarnf = this.cwrap('lua_setwarnf', null, ['number', 'number', 'number']);
            this.lua_warning = this.cwrap('lua_warning', null, ['number', 'string', 'number']);
            this.lua_error = this.cwrap('lua_error', 'number', ['number']);
            this.lua_next = this.cwrap('lua_next', 'number', ['number', 'number']);
            this.lua_concat = this.cwrap('lua_concat', null, ['number', 'number']);
            this.lua_len = this.cwrap('lua_len', null, ['number', 'number']);
            this.lua_stringtonumber = this.cwrap('lua_stringtonumber', 'number', ['number', 'string']);
            this.lua_getallocf = this.cwrap('lua_getallocf', 'number', ['number', 'number']);
            this.lua_setallocf = this.cwrap('lua_setallocf', null, ['number', 'number', 'number']);
            this.lua_toclose = this.cwrap('lua_toclose', null, ['number', 'number']);
            this.lua_closeslot = this.cwrap('lua_closeslot', null, ['number', 'number']);
            this.lua_getstack = this.cwrap('lua_getstack', 'number', ['number', 'number', 'number']);
            this.lua_getinfo = this.cwrap('lua_getinfo', 'number', ['number', 'string', 'number']);
            this.lua_getlocal = this.cwrap('lua_getlocal', 'string', ['number', 'number', 'number']);
            this.lua_setlocal = this.cwrap('lua_setlocal', 'string', ['number', 'number', 'number']);
            this.lua_getupvalue = this.cwrap('lua_getupvalue', 'string', ['number', 'number', 'number']);
            this.lua_setupvalue = this.cwrap('lua_setupvalue', 'string', ['number', 'number', 'number']);
            this.lua_upvalueid = this.cwrap('lua_upvalueid', 'number', ['number', 'number', 'number']);
            this.lua_upvaluejoin = this.cwrap('lua_upvaluejoin', null, ['number', 'number', 'number', 'number', 'number']);
            this.lua_sethook = this.cwrap('lua_sethook', null, ['number', 'number', 'number', 'number']);
            this.lua_gethook = this.cwrap('lua_gethook', 'number', ['number']);
            this.lua_gethookmask = this.cwrap('lua_gethookmask', 'number', ['number']);
            this.lua_gethookcount = this.cwrap('lua_gethookcount', 'number', ['number']);
            this.lua_setcstacklimit = this.cwrap('lua_setcstacklimit', 'number', ['number', 'number']);
            this.luaopen_base = this.cwrap('luaopen_base', 'number', ['number']);
            this.luaopen_coroutine = this.cwrap('luaopen_coroutine', 'number', ['number']);
            this.luaopen_table = this.cwrap('luaopen_table', 'number', ['number']);
            this.luaopen_io = this.cwrap('luaopen_io', 'number', ['number']);
            this.luaopen_os = this.cwrap('luaopen_os', 'number', ['number']);
            this.luaopen_string = this.cwrap('luaopen_string', 'number', ['number']);
            this.luaopen_utf8 = this.cwrap('luaopen_utf8', 'number', ['number']);
            this.luaopen_math = this.cwrap('luaopen_math', 'number', ['number']);
            this.luaopen_debug = this.cwrap('luaopen_debug', 'number', ['number']);
            this.luaopen_package = this.cwrap('luaopen_package', 'number', ['number']);
            this.luaL_openlibs = this.cwrap('luaL_openlibs', null, ['number']);
        }
        lua_remove(luaState, index) {
            this.lua_rotate(luaState, index, -1);
            this.lua_pop(luaState, 1);
        }
        lua_pop(luaState, count) {
            this.lua_settop(luaState, -count - 1);
        }
        luaL_getmetatable(luaState, name) {
            return this.lua_getfield(luaState, LUA_REGISTRYINDEX, name);
        }
        lua_yield(luaState, count) {
            return this.lua_yieldk(luaState, count, 0, null);
        }
        lua_upvalueindex(index) {
            return LUA_REGISTRYINDEX - index;
        }
        ref(data) {
            const existing = this.referenceTracker.get(data);
            if (existing) {
                existing.refCount++;
                return existing.index;
            }
            const availableIndex = this.availableReferences.pop();
            const index = availableIndex === undefined ? this.referenceMap.size + 1 : availableIndex;
            this.referenceMap.set(index, data);
            this.referenceTracker.set(data, {
                refCount: 1,
                index,
            });
            this.lastRefIndex = index;
            return index;
        }
        unref(index) {
            const ref = this.referenceMap.get(index);
            if (ref === undefined) {
                return;
            }
            const metadata = this.referenceTracker.get(ref);
            if (metadata === undefined) {
                this.referenceTracker.delete(ref);
                this.availableReferences.push(index);
                return;
            }
            metadata.refCount--;
            if (metadata.refCount <= 0) {
                this.referenceTracker.delete(ref);
                this.referenceMap.delete(index);
                this.availableReferences.push(index);
            }
        }
        getRef(index) {
            return this.referenceMap.get(index);
        }
        getLastRefIndex() {
            return this.lastRefIndex;
        }
        printRefs() {
            for (const [key, value] of this.referenceMap.entries()) {
                console.log(key, value);
            }
        }
        cwrap(name, returnType, argTypes) {
            const hasStringOrNumber = argTypes.some((argType) => argType === 'string|number');
            if (!hasStringOrNumber) {
                return (...args) => this.module.ccall(name, returnType, argTypes, args);
            }
            return (...args) => {
                const pointersToBeFreed = [];
                const resolvedArgTypes = argTypes.map((argType, i) => {
                    var _a;
                    if (argType === 'string|number') {
                        if (typeof args[i] === 'number') {
                            return 'number';
                        }
                        else {
                            if (((_a = args[i]) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
                                const bufferPointer = this.module.stringToNewUTF8(args[i]);
                                args[i] = bufferPointer;
                                pointersToBeFreed.push(bufferPointer);
                                return 'number';
                            }
                            else {
                                return 'string';
                            }
                        }
                    }
                    return argType;
                });
                try {
                    return this.module.ccall(name, returnType, resolvedArgTypes, args);
                }
                finally {
                    for (const pointer of pointersToBeFreed) {
                        this.module._free(pointer);
                    }
                }
            };
        }
    }

    var version = '1.15.0';

    class LuaFactory {
        constructor(customWasmUri, environmentVariables) {
            var _a;
            if (customWasmUri === undefined) {
                const isBrowser = (typeof window === 'object' && typeof window.document !== 'undefined') ||
                    (typeof self === 'object' && ((_a = self === null || self === void 0 ? void 0 : self.constructor) === null || _a === void 0 ? void 0 : _a.name) === 'DedicatedWorkerGlobalScope');
                if (isBrowser) {
                    const majorminor = version.slice(0, version.lastIndexOf('.'));
                    customWasmUri = `https://unpkg.com/wasmoon@${majorminor}/dist/glue.wasm`;
                }
            }
            this.luaWasmPromise = LuaWasm.initialize(customWasmUri, environmentVariables);
        }
        async mountFile(path, content) {
            this.mountFileSync(await this.getLuaModule(), path, content);
        }
        mountFileSync(luaWasm, path, content) {
            const fileSep = path.lastIndexOf('/');
            const file = path.substring(fileSep + 1);
            const body = path.substring(0, path.length - file.length - 1);
            if (body.length > 0) {
                const parts = body.split('/').reverse();
                let parent = '';
                while (parts.length) {
                    const part = parts.pop();
                    if (!part) {
                        continue;
                    }
                    const current = `${parent}/${part}`;
                    try {
                        luaWasm.module.FS.mkdir(current);
                    }
                    catch (err) {
                    }
                    parent = current;
                }
            }
            luaWasm.module.FS.writeFile(path, content);
        }
        async createEngine(options = {}) {
            return new LuaEngine(await this.getLuaModule(), options);
        }
        async getLuaModule() {
            return this.luaWasmPromise;
        }
    }

    exports.Decoration = Decoration;
    exports.LUAI_MAXSTACK = LUAI_MAXSTACK;
    exports.LUA_MULTRET = LUA_MULTRET;
    exports.LUA_REGISTRYINDEX = LUA_REGISTRYINDEX;
    exports.LuaEngine = LuaEngine;
    exports.LuaFactory = LuaFactory;
    exports.LuaGlobal = Global;
    exports.LuaMultiReturn = MultiReturn;
    exports.LuaRawResult = RawResult;
    exports.LuaThread = Thread;
    exports.LuaTimeoutError = LuaTimeoutError;
    exports.LuaTypeExtension = LuaTypeExtension;
    exports.LuaWasm = LuaWasm;
    exports.PointerSize = PointerSize;
    exports.decorate = decorate;
    exports.decorateFunction = decorateFunction;
    exports.decorateProxy = decorateProxy;
    exports.decorateUserdata = decorateUserdata;

}));
