local WorkerManager = require('workers')
local webview = require('webview')
local base64 = require("base64")

local b64 = base64.b64

-- Class declarations
local RInstance = class()
local RRepl = class()

local POOL_SIZE_INITIAL = 4
local POOL_SIZE_MINIMUM = 2

-- Create JS environment & channel manager
local wv = webview()
local wm = WorkerManager(6767, wv)

local workerPool = {}
local function initThreadPool(num)
	for i=1,num do
		-- Add new worker to pool
		local worker = wm:newWorker()
		table.insert(workerPool, worker)
	end
end
initThreadPool(POOL_SIZE_INITIAL)

local function getWorker()
	-- Get from workerPool
	local w = table.remove(workerPool)
	
	-- Add new worker to the pool
	-- TODO: This needs to be nonblocking
	if #workerPool < POOL_SIZE_MINIMUM then
		local worker = wm:newWorker()
		table.insert(workerPool, worker)
	end
	
	return w
end

function RInstance:init(source, ...)
	-- Get a worker instance
	self._worker = getWorker()
	
	source = source or ""
	assert(type(source) == "string")

	local workerSource = [[
		// Download WebR runtime
		const { WebR, ChannelType } = await import('https://webr.r-wasm.org/latest/webr.mjs');
		
		// Initialise WebR
		let webR;
		try {
			webR = new WebR({
				channelType: ChannelType.PostMessage,
			});
			await webR.init();
		} catch(e) {
			error(e.message);
		}
		
		// Set default render device
		await webR.evalRVoid('options(device=webr::canvas)');
		
		// Load the R code
		const code = await codea.recv();
		await webR.evalR(code);
		
		// Signal to Codea that the instance is ready
		codea.send("_ready_");
		
		// Restructures RObject js results into Codea friendly structures
		// for transfer back to Codea.
		const transferableResult = (r)=>{
			if (typeof r == "object") {
				if (r.names != null) {
					let o = new Object();
					let i = 0;
					for (const key of r.names) {
						o[key] = transferableResult(r.values[i++]);
					}
					return o;
				} else {
					let a = new Array();
					for (const val of r.values) {
						a.push(transferableResult(val));
					}
					if (a.length == 1) {
						return a[0];
					}
					return a;
				}
			}
			return r;
		};
		
		// Our plot canvas
		let gfxReceiverId;
		let canvas = new OffscreenCanvas(1008, 1008);
		let isNewCanvas = false;
		
		// Async message handler
		(async ()=>{
			while (true) {
				const output = await webR.read();
				switch(output.type) {
					case "stdout":
						//print(output.data);
						break;
					case "stderr":
						error(output.data);
						break;
					case "prompt":
						//print(">");
						break;
					case "canvas":
						try{
						if (output.data.event === 'canvasImage') {
							const width = output.data.image.width;
							const height = output.data.image.height;
							
							// Is the canvas the correct size
							// or should we create a new one anyway?
							if (canvas.width != width ||
								canvas.height != height || isNewCanvas == true)
							{
								canvas = new OffscreenCanvas(width, height);
								isNewCanvas = true;
							}
							
							// Draw to our canvas
							canvas.getContext('2d').drawImage(output.data.image, 0, 0);
							
							// Prepare and send our updated image
							let data = await canvas.convertToBlob();
							data = await data.arrayBuffer();
							data = new Uint8Array(data);
							
							codea.send({
								data: {
									data: data,
									isNewPlot: isNewCanvas
								},
								id: gfxReceiverId
							});
							isNewCanvas = false;
						
						} else if (output.data.event === 'canvasNewPage') {
							isNewCanvas = true;
						}
						}catch(e){
							error(e.message);
						}
						break;
					default:
						print("R message:", output.type);
						break;
				}
			}
		})();
		
		// Main invocation loop
		while (true) {
			const invocation = await codea.recv();
			
			// Update the graphics receiver
			gfxReceiverId = invocation.gfxId;
			
			let result;
			try {
				const fn = await webR.evalR(invocation.fn);
				result = await fn(...invocation.args);
				//print(typeof result, invocation.fn);
			} catch(e) {
				error(e.message);
			}
			
			// Restructure the result into an object suitable for Codea
			const tresult = transferableResult(result);
			
			// Useful debuging stuff
			//print(JSON.stringify(result));
			//print(JSON.stringify(tresult));
			
			codea.send({
				data: tresult,
				id: invocation.id
			});
		}
	]]
	
	-- Send the worker code
	self._worker:send({
		code = workerSource,
		args = {}
	});
	
	-- Send the R code
	self._worker:send(source);
	
	-- Wait for 'ready' signal
	print("Initialising WebR.\nThis could take a moment...")
	assert(self._worker:recv() == "_ready_")
	print("WebR Initialised.")
	
	-- Each call is assigned an ID so we can separate results
	local callId = 1
	
	setmetatable(self, {
		__index = function(t, k)
			rawset(t, k, function(...)
				self._worker:send({
					fn = k,
					args = {...},
					id = callId,
					gfxId = callId + 1
				});
				
				-- Create new receiver objects for this call
				local receiver = self._worker:newReceiver(callId)
				local gfxReceiver = self._worker:newReceiver(callId + 1)
				
				-- Auto hook up the gfxReceiver
				gfxReceiver:next(function(msg)
					if receiver.onplot then
						receiver.onplot(image(msg.data), msg.isNewPlot)
					end
				end)
				
				-- Increment callId
				callId = callId + 2
				
				return receiver
			end)
			return t[k]
		end
	})
	
	-- Add eval function (Work-In-Progress)
	self.eval = function(self, expr)
		local r = self["webr::eval_r"](expr)
		--print(r)
		return r:result()
	end
end

function RRepl:init(source)
	-- Get a worker instance
	self._worker = getWorker()
	
	source = source or "0"
	assert(type(source) == "string")

	local workerSource = [[
		// Download WebR runtime
		const { WebR, ChannelType } = await import('https://webr.r-wasm.org/latest/webr.mjs');
		
		// Initialise WebR
		let webR;
		try {
			webR = new WebR({
				channelType: ChannelType.PostMessage,
			});
			await webR.init();
		} catch(e) {
			error(e.message);
		}
		
		// Set default render device
		await webR.evalRVoid('options(device=webr::canvas)');
		
		// Load the R code
		const code = await codea.recv();
		await webR.evalR(code);
		
		// Restructures RObject js results into Codea friendly structures
		// for transfer back to Codea.
		const transferableResult = (r)=>{
			if (typeof r == "object") {
				if (r.names != null) {
					let o = new Object();
					let i = 0;
					for (const key of r.names) {
						o[key] = transferableResult(r.values[i++]);
					}
					return o;
				} else {
					let a = new Array();
					for (const val of r.values) {
						a.push(transferableResult(val));
					}
					if (a.length == 1) {
						return a[0];
					}
					return a;
				}
			}
			return r;
		};
		
		// Our plot canvas
		const gfxReceiverId = 1;
		let canvas = new OffscreenCanvas(1008, 1008);
		let isNewCanvas = false;
		
		// Async message handler
		(async ()=>{
			while (true) {
				const output = await webR.read();
				print(output.data);
				switch(output.type) {
					case "stdout":
						//print(output.data);
						break;
					case "stderr":
						print(output.data);
						break;
					case "prompt":
						//print(">");
						break;
					case "canvas":
						if (output.data.event === 'canvasImage') {
							print("draw");
							const width = output.data.image.width;
							const height = output.data.image.height;
							
							// Is the canvas the correct size
							// or should we create a new one anyway?
							if (canvas.width != width ||
								canvas.height != height || isNewCanvas == true)
							{
								canvas = new OffscreenCanvas(width, height);
								isNewCanvas = true;
							}
							
							// Draw to our canvas
							canvas.getContext('2d').drawImage(output.data.image, 0, 0);
							
							// Prepare and send our updated image
							let data = await canvas.convertToBlob();
							data = await data.arrayBuffer();
							data = new Uint8Array(data);
							
							codea.send({
								data: {
									data: data,
									isNewPlot: isNewCanvas
								},
								id: gfxReceiverId
							});
							isNewCanvas = false;
						
						} else if (output.data.event === 'canvasNewPage') {
							isNewCanvas = true;
							print("newCanvas");
						}
						break;
					default:
						print("R message:", output.type);
						break;
				}
			}
		})();
		
		// Signal to Codea that the instance is ready
		codea.send("_ready_");
		
		// Main invocation loop
		while (true) {
			const code = await codea.recv();
			
			let result;
			try {
				result = await webR.evalR(code);
				result = await result.toJs();
			} catch(e) {
				error(e.message);
			}
			
			// Restructure the result into an object suitable for Codea
			const tresult = transferableResult(result);
			
			codea.send({
				data: tresult,
				id: 2
			});
		}
	]]
	
	-- Send the worker code
	self._worker:send({
		code = workerSource,
		args = {}
	});
	
	-- Send the R code
	self._worker:send(source);
	
	-- Wait for 'ready' signal
	print("Initialising WebR.\nThis could take a moment...")
	assert(self._worker:recv() == "_ready_")
	print("WebR Initialised.")
	
	-- Reuse the receivers
	self.gfxReceiver = self._worker:newReceiver(1)
	self.receiver = self._worker:newReceiver(2)
	
	-- Print callback
	self.receiver:next(function(r)
		if type(r) == "table" then
			print(json.encode(r))
		else
			print(r)
		end
	end)
end

function RRepl:eval(code)
	self._worker:send(code); -- Send code to be evaluated
end

local exports = setmetatable({}, {
	__call = function(t, source)
		return RInstance(source)
	end
})

-- Repl instance (Work-In-Progress)
exports.repl = function(source)
	local repl = RRepl(source)
	
	local gfx = nil
	repl.gfxReceiver:next(function(msg)
		print("gfx")
		gfx = image(msg.data)
	end)
	
	_G.draw = function()
		if gfx then
			background(255)
			local size = math.min(WIDTH, HEIGHT)
			spriteMode(CENTER)
			sprite(gfx, WIDTH/2, HEIGHT/2, size, size)
		end
	end
	
	return function(code)
		print(code) -- echo
		repl:eval(code)
	end
end

-- Add automated update function
exports.updatePerFrameLimit = 256
local _tweenup = tween.update
tween.update = function(...)
	_tweenup(...)
	wm:update(limit or exports.updatePerFrameLimit)
end

-- Set global
return {
	R = exports
}