local Impl = {
	config = {
		projectName = "R",
		
		workerPoolInitial = 4,
		workerPoolMinimum = 2,
		
		connectionPort = 6767
	},
	auto = {}
}

local workerBaseScript = readBlob("impl/worker.js")
local workerInstanceScript = workerBaseScript .. readBlob("impl/worker_instance.js")
local workerReplScript = workerBaseScript .. readBlob("impl/worker_repl.js")

local Instance = class()
function Instance:init(source)
	self.worker = Impl.auto.getWorker()
	
	source = source or "0"
	assert(type(source) == "string")
	
	-- Load the worker script
	self.worker:send({
		code = workerInstanceScript,
		args = {}
	});
	
	-- Load the R code
	self.worker:send(source);
	
	-- Wait for 'ready' signal
	print("Initialising WebR.\nThis could take a moment...")
	assert(self.worker:recv() == "_ready_")
	print("WebR Initialised.")
	
	-- Each call is assigned an ID so we can separate results
	local callId = 1
	setmetatable(self, {
		__index = function(t, k)
			rawset(t, k, function(...)
				self.worker:send({
					fn = k,
					args = {...},
					id = callId,
					gfxId = callId + 1
				});
				
				-- Create new receiver objects for this call
				local receiver = self.worker:newReceiver(callId)
				local gfxReceiver = self.worker:newReceiver(callId + 1)
				
				local plotFn
				function receiver:onplot(fn)
					plotFn = fn
				end
				
				-- Auto hook up the gfxReceiver
				gfxReceiver:next(function(msg)
					if plotFn then
						plotFn(image(msg.data), msg.isNewPlot)
					end
				end)
				
				-- Increment callId
				callId = callId + 2
				
				return receiver
			end)
			return t[k]
		end
	})
end

local REPL = class()
function REPL:init(source)
	self.worker = Impl.auto.getWorker()
	
	source = source or "0"
	assert(type(source) == "string")
	
	-- Load the worker script
	self.worker:send({
		code = workerReplScript,
		args = {}
	});
	
	-- Load the R code
	self.worker:send(source);
	
	-- Reuse the receivers
	self.gfxReceiver = self.worker:newReceiver(1)
	self.receiver = self.worker:newReceiver(2)
	
	-- Setup new draw function to display R graphics
	local gfx = nil
	self.gfxReceiver:next(function(msg)
		gfx = image(msg.data)
	end)
	_G.draw = function()
		if gfx then
			background(255)
			local size = math.min(WIDTH, HEIGHT)
			spriteMode(CENTER)
			sprite(gfx, WIDTH/2, HEIGHT/2, size, size)
		end
	end
	
	-- Wait for 'ready' signal
	print("Initialising WebR.\nThis could take a moment...")
	assert(self.worker:recv() == "_ready_")
	print("WebR Initialised.")
	
	output.clear()
	print([[-- R REPL --
Enter commands using 'R(<r_code_string>)']])
	print([[Examples:
	- R("c(10,9,8)")
	- R("plot2d(500)")]])
	
	setmetatable(self, {
		__call = function(_,expr)
			print("> " .. expr) -- echo
			self.worker:send(expr); -- Send code to be evaluated
			self.receiver.hasResult = false
			local r = self.receiver:waitForResult()
			if type(r) == "table" then
				r = json.encode(r)
			end
			print("-> ", r)
			return r
		end
	})
end

-- Export our functions
Impl.exports = {
	default = Instance,
	instance = Instance,
	repl = REPL
}

return Impl