local Impl = {
	config = {
		projectName = "R",
		
		workerPoolInitial = 4,
		workerPoolMinimum = 2,
		
		connectionPort = 6767
	},
	auto = {}
}

local workerScript = readBlob("impl/worker.js")
local Rbuiltin = readBlob("impl/builtin.r")

local Graphics = class()
function Graphics:init(worker)
	self.receiver = worker:newReceiver()
	
	self.images = {}
	self.receiver:next(function(msg)
		local img = image(msg.data)
		local new = msg.isNewPlot
		if new then
			-- New image
			table.insert(self.images, img)
		else
			-- Update image
			self.images[#self.images] = img
		end
		-- Call current graphics callback
		self.currentFn(img, new)
	end)
	
	self.defaultFn = function(img, new) end
	self:setDefault()
end
function Graphics:setDefault()
	self.currentFn = self.defaultFn
	_G.draw = function()
		background(255)
		local img = self.images[#self.images]
		if img then
			local size = math.min(WIDTH, HEIGHT)
			spriteMode(CENTER)
			sprite(img, WIDTH/2, HEIGHT/2, size, size)
		end
	end
end
function Graphics:clear()
	self.images = {}
end
function Graphics:setFn(fn, drawFn)
	self.currentFn = fn
	-- Call for any images we may have missed
	for _,img in ipairs(self.images) do
		fn(img, true)
	end
	
	if drawFn then
		_G.draw = drawFn
		drawFn()
	end
end

local RtoLua
RtoLua = function(obj, worker, gfx)
	--print("RtoLua", json.encode(obj))
	
	if type(obj) == "table" then
		if obj.fn then
			return function(...)
				-- Create new receiver objects for this call
				local receiver = worker:newReceiver()
				
				-- Pre-process received results
				receiver._resultProcessor = function(result)
					return RtoLua(result, worker)
				end
				
				-- Send call
				worker:send({
					op = "call",
					fn = obj.fn,
					args = {...},
					id = receiver.id
				});
				
				if obj.async then
					return receiver
				else
					return receiver:waitForResult()
				end
			end
		else
			if obj.names then
				local nobj = {}
				for i,name in ipairs(obj.names) do
					nobj[name] = RtoLua(obj.values[i])
				end
				return nobj
			elseif obj.values then
				if #obj.values == 1 then
					return RtoLua(obj.values[1])
				end
				return RtoLua(obj.values)
			else
				local nobj = {}
				for i, v in ipairs(obj) do
					nobj[i] = RtoLua(v)
				end
				return nobj
			end
		end
	else
		return obj
	end
end

local Instance = class()
function Instance:init(source)
	self.worker = Impl.auto.getWorker()
	self.gfx = Graphics(self.worker)
	
	source = source or "0"
	assert(type(source) == "string")
	
	-- Load the worker script
	self.worker:send({
		code = workerScript,
		args = {}
	});
	
	-- Load the R code
	self.worker:send(Rbuiltin .. "\n\n" .. source);
	
	-- Wait for 'ready' signal
	print("Initialising WebR.\nThis could take a moment...")
	local msg = self.worker:recv()
	assert(msg == "_ready_")
	print("WebR Initialised.")
	
	-- Set the graphics receiver
	self.worker:send({
		op = "setGfxReceiver",
		gfxId = self.gfx.receiver.id
	});
	
	self.eval = function(expr, levelOffset)
		levelOffset = levelOffset or 0 -- default
	
		-- Create a new receiver and begin R evaluation
		local receiver = self.worker:newReceiver()	
			
		-- Add caller info to the receiver so we can track the source of errors
		-- offset of -1 means, no caller info
		if levelOffset ~= -1 then
			local info = debug.getinfo(2 + levelOffset, "Sl")
			receiver.caller = info.short_src .. ":" ..info.currentline
		end
		
		self.worker:send({
			op = "read",
			key = expr,
			id = receiver.id
		});
		
		-- Wait for the result from R
		local result = receiver:waitForResult()
		
		-- Convert to a useful format for Lua
		result = RtoLua(result, self.worker, self.gfx)
		return result
	end
	
	self.asLuaDF = setmetatable({}, {
		__index = function(_, k)
			local columns = self[k]
			local rowNames = self["row.names(" .. k .. ")"]
			local rowIndex = {}
			local rows = {}
			for i,name in ipairs(rowNames) do
				local row = {}
				rows[i] = row
				rows[name] = row
				rowIndex[name] = i
				for colName, values in pairs(columns) do
					row[colName] = values[i]
				end
			end
			
			-- Add helper functions
			return setmetatable(rows, {
				__index = {
					_nameOf = function(index)
						return rowNames[index]
					end,
					_indexOf = function(rowName)
						return rowIndex[rowName]
					end
				}
			})
		end
	})
	
	self.repl = function()
		print([[-- R REPL --
Enter commands using 'R(<r_code_string>)']])
		print([[Examples:
		- R("c(10,9,8)")
		- R("plot2d(500)")]])
		
		return function(expr, silent)
			if not silent then print("> " .. expr) end -- echo
			local r = self.eval(expr, -1)
			if type(r) == "table" then
				r = json.encode(r)
			end
			if not silent then print("-> ", r) end
		end
	end
	
	self.installPackages = function(...)
		self["webr::install"]({...})
	end
	
	setmetatable(self, {
		__index = function(t, k)
			-- Is this expected to be an async function call?
			local isAsync = false
			if k:sub(1,6) == "async_" then
				k = k:sub(7)
				isAsync = true
			end
			
			-- Create a new receiver
			local receiver = self.worker:newReceiver()
			
			-- Add caller info to the receiver so we can track the source of errors
			local info = debug.getinfo(2, "Sl")
			receiver.caller = info.short_src .. ":" ..info.currentline
			
			-- Begin R evaluation
			self.worker:send({
				op = "read",
				key = k,
				id = receiver.id
			});
			
			-- Wait for the result from R
			local result = receiver:waitForResult()
			
			-- Mark this result as 'async' in case the result is a function.
			-- This acts as a flag for the function wrapper
			-- in RtoLua.
			result.async = isAsync
			
			-- Convert to a useful format for Lua
			result = RtoLua(result, self.worker, self.gfx)
			return result
		end,
		__newindex = function(t, k, v)
			local receiver = self.worker:newReceiver()
			
			-- Add caller info to the receiver so we can track the source of errors
			local info = debug.getinfo(2, "Sl")
			receiver.caller = info.short_src .. ":" ..info.currentline
			
			self.worker:send({
				op = "write",
				key = k,
				value = v,
				id = receiver.id
			});
			receiver:waitForResult()
		end,
		__call = function(_, expr)
			return self.eval(expr, 1)
		end
	});
end

-- Export our functions
Impl.exports = {
	default = Instance,
	instance = Instance,
}

return Impl