// Download WebR runtime
const { WebR, WebRDataJsNode, ChannelType, isRFunction } = await import('https://webr.r-wasm.org/latest/webr.mjs');

// Initialise WebR
let webR;
try {
	webR = new WebR({
		channelType: ChannelType.PostMessage,
	});
	await webR.init();
} catch(e) {
	error(e.message);
}

// Set default render device
await webR.evalRVoid('options(device=webr::canvas)');

// Load the R code
const code = await codea.recv();
await webR.evalR(code);

// Our plot canvas
let gfxReceiverId = -1;
let canvas = new OffscreenCanvas(1008, 1008);
let isNewCanvas = false;
let seenPrompt = false;

// Async message handler
(async ()=>{
	while (true) {
		const output = await webR.read();
		switch(output.type) {
			case "stdout":
				if (seenPrompt === true) { print(output.data); }
				break;
			case "stderr":
				if (seenPrompt === true) { warn(output.data); }
				break;
			case "prompt":
				//print(output.data);
				seenPrompt = true;
				break;
			case "canvas":
				if (gfxReceiverId === -1) {
					break;
				}
			
				if (output.data.event === 'canvasImage') {
					const width = output.data.image.width;
					const height = output.data.image.height;
					
					// Is the canvas the correct size
					// or should we create a new one anyway?
					if (canvas.width != width ||
						canvas.height != height || isNewCanvas == true)
					{
						canvas = new OffscreenCanvas(width, height);
						isNewCanvas = true;
					}
					
					// Draw to our canvas
					canvas.getContext('2d').drawImage(output.data.image, 0, 0);
					
					// Prepare and send our updated image
					let data = await canvas.convertToBlob();
					data = await data.arrayBuffer();
					data = new Uint8Array(data);
					
					codea.send({
						data: {
							data: data,
							isNewPlot: isNewCanvas
						},
						id: gfxReceiverId
					});
					isNewCanvas = false;
				
				} else if (output.data.event === 'canvasNewPage') {
					isNewCanvas = true;
				}
				break;
			default:
				print("R message:", output.type);
				break;
		}
	}
})();

// Create shelter
const R = await new webR.Shelter();

// Signal to Codea that the instance is ready
codea.send("_ready_");

// Main invocation loop
while (true) {
	const inv = await codea.recv();
	let result;
	switch(inv.op) {
		case "read":
		{
			try {
				result = await R.evalR(inv.key);
				
				if (isRFunction(result)) {
					codea.send({
						data: {
							fn: inv.key
						},
						id: inv.id
					});
				} else {
					result = await result.toJs();
					codea.send({
						data: result,
						id: inv.id
					});
				}
			} catch(e) {
				error("[R] Error: " + e.message, inv.id);
			} finally {
				R.purge()
			}
		}
		break;
		case "write":
		{
			try {
				let fn = await R.evalR("function(v) {" + inv.key + " <<- v }");
				
				let value = inv.value;
				if (typeof value === "object") {
					//print(JSON.stringify(inv.value));
					if (Array.isArray(inv.value) &&
						inv.value.every((val, i, arr) => typeof val === typeof inv.value[0])) {
						value = await new R.RObject(inv.value);
					} else {
						value = await new R.RList(inv.value);
					}
					//print(JSON.stringify(await value.toJs()));
				}
				fn.exec(value);
				
				// Signal that the write is complete
				codea.send({
					data: null,
					id: inv.id
				});
			} catch(e) {
				error("[R] Error: " + e.message, inv.id);
			} finally {
				R.purge()
			}
		}
		break;
		case "call":
		{
			try {
				let fn = await R.evalR(inv.fn);
				if (isRFunction(fn)) {
					result = await fn(...inv.args);
					if (isRFunction(result)) {
						warn("Unable to return functions currently!");
					} else {
						codea.send({
							data: result,
							id: inv.id
						});
					}
				} else {
					error("Attempt to call non function value: " + inv.fn);
				}
			} catch(e) {
				error("[R] Error: " + e.message, inv.id);
			} finally {
				R.purge()
			}
		}
		break;
		case "setGfxReceiver":
		{
			gfxReceiverId = inv.gfxId;
		}
		break;
	}
}