function setup()
    viewer.mode = FULLSCREEN

    playerX = WIDTH/2
    playerY = HEIGHT/2
    
    tileSize = 64
    
    commands = {
        ["Go up"] = function() movePlayer(0, 1) end,
        ["Go down"] = function() movePlayer(0, -1) end,
        ["Go left"] = function() movePlayer(-1, 0) end,
        ["Go right"] = function() movePlayer(1, 0) end,
        ["Say hello"] = function() say("Hello!") end,
        ["Say goodbye"] = function() say("Goodbye!") end,
    }

    maxRecognitionTime = 0.5
    lastFailureTime = -1
    lastSayTime = -1
    lastSay = ""
    messageStart = -1
    shouldStartListening = false
    authorized = false
    
    objc.SFSpeechRecognizer:requestAuthorization_(function(iStatus)
        if iStatus == objc.enum.SFSpeechRecognizerAuthorizationStatus.authorized then
            locale = objc.NSLocale:localeWithLocaleIdentifier_("en_US")
            audioEngine = objc.AVAudioEngine()
            audioSession = objc.AVAudioSession.sharedInstance
            audioSession:setCategory_mode_options_error_("AVAudioSessionCategoryRecord", "AVAudioSessionModeMeasurement", 2, nil)
            audioSession:setActive_withOptions_error_(true, 1, nil)
            speechRecognizer = objc.SFSpeechRecognizer:alloc():initWithLocale_(locale)
            shouldStartListening = true
            authorized = true
        end
    end)
end

function startListening()
    recognitionRequest = objc.SFSpeechAudioBufferRecognitionRequest()
    recognitionRequest.shouldReportPartialResults = true
    recognitionRequest.requiresOnDeviceRecognition = true
    recognitionTask = speechRecognizer:recognitionTaskWithRequest_resultHandler_(recognitionRequest,
    function (oResult, oError)
        if oResult ~= nil and oResult.bestTranscription.formattedString ~= nil then
            if messageStart == -1 then
                messageStart = ElapsedTime
            end
            local message = oResult.bestTranscription.formattedString
            local foundCommand = false
            for k, v in pairs(commands) do
                if message == k then
                    v()
                    foundCommand = true
                    restartListening()
                    break
                end
            end
        end
    end)
    inputNode = audioEngine.inputNode
    recordingFormat = inputNode:outputFormatForBus_(0)
    inputNode:installTapOnBus_bufferSize_format_block_(0, 1024, recordingFormat,
        function(oBuffer, oTime) recognitionRequest:appendAudioPCMBuffer_(oBuffer) end)
    audioEngine:prepare()
    audioEngine:startAndReturnError_(nil)
end

function restartListening()
    messageStart = -1
    stopAudio()
    shouldStartListening = true
end

function stopAudio()
    if audioEngine then
        if recognitionTask ~= nil then
            -- Note that this will cause the recognitionTask to invoke the
            -- callback with an oError indicating it has been cancelled.
            recognitionTask:cancel()
        end
        inputNode:removeTapOnBus_(0)
        audioEngine:stop()
        recognitionRequest = nil
        recognitionTask = nil
    end
end

function willClose()
    stopAudio()
end

function movePlayer(x, y)
    playerX = playerX + x * tileSize
    playerY = playerY + y * tileSize
end

function say(message)
    lastSayTime = ElapsedTime
    lastSay = message
end

function update()
    if shouldStartListening then
        startListening()
        shouldStartListening = false
    end
    
    if messageStart > 0 and ElapsedTime - messageStart > maxRecognitionTime then
        lastFailureTime = ElapsedTime
        restartListening()
    end
end

function draw()
    update()
    
    background(40, 40, 50)

    fill(255)
    spriteMode(CENTER)    
    sprite(asset.builtin.Planet_Cute.Character_Boy,playerX,playerY)
    
    if lastFailureTime > 0 and ElapsedTime - lastFailureTime < 1 then
        sprite(asset.builtin.Planet_Cute.SpeechBubble, playerX + 100, playerY + 100)
        textMode(CENTER)
        fill(0)
        fontSize(64)
        text("?", playerX + 100, playerY + 70)
    end

    if lastSayTime > 0 and ElapsedTime - lastSayTime < 1 then
        sprite(asset.builtin.Planet_Cute.SpeechBubble, playerX + 100, playerY + 100)
        textMode(CENTER)
        fill(0)
        fontSize(12)
        text(lastSay, playerX + 100, playerY + 70)
    end
end
